package com.bxm.localnews.admin.service.medal.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.admin.config.ClientRouteProperties;
import com.bxm.localnews.admin.config.H5JumpAddressProperties;
import com.bxm.localnews.admin.domain.medal.UserMedalGrantInfoMapper;
import com.bxm.localnews.admin.domain.medal.UserMedalInfoMapper;
import com.bxm.localnews.admin.domain.medal.UserMedalRelationInfoMapper;
import com.bxm.localnews.admin.dto.UserGrantMedalDTO;
import com.bxm.localnews.admin.entity.medal.UserMedalGrantInfoEntity;
import com.bxm.localnews.admin.entity.medal.UserMedalInfoEntity;
import com.bxm.localnews.admin.entity.medal.UserMedalRelationInfoEntity;
import com.bxm.localnews.admin.enums.MedalCategoryEnum;
import com.bxm.localnews.admin.enums.MedalGrantStatusEnum;
import com.bxm.localnews.admin.param.medal.ConfirmRecordParam;
import com.bxm.localnews.admin.param.medal.MedalGrantQueryParam;
import com.bxm.localnews.admin.param.medal.SaveCustomMedalGrantParam;
import com.bxm.localnews.admin.service.medal.MedalGrantManageService;
import com.bxm.localnews.admin.service.medal.cache.UserMedalPopCacheManager;
import com.bxm.localnews.admin.utils.SecurityUtils;
import com.bxm.localnews.admin.vo.medal.CustomGrantDetailVO;
import com.bxm.localnews.admin.vo.medal.MedalGrantListVO;
import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.web.util.UriUtils;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/3/2 3:49 下午
 */
@Slf4j
@Service
@AllArgsConstructor
public class MedalGrantManageServiceImpl implements MedalGrantManageService {

    private UserMedalGrantInfoMapper userMedalGrantInfoMapper;

    private UserMedalRelationInfoMapper userMedalRelationInfoMapper;

    private UserMedalInfoMapper userMedalInfoMapper;

    private SequenceCreater sequenceCreater;

    private UserMedalPopCacheManager userMedalPopCacheManager;

    private H5JumpAddressProperties h5JumpAddressProperties;

    private DomainIntegrationService domainIntegrationService;

    private ClientRouteProperties clientRouteProperties;

    @Override
    public PageWarper<MedalGrantListVO> getGrantList(MedalGrantQueryParam param) {
        return new PageWarper<>(userMedalGrantInfoMapper.getGrantList(param));
    }

    @Override
    public Message saveOrUpdateGrant(SaveCustomMedalGrantParam param) {
        if (CollectionUtils.isEmpty(param.getGrantUserIds())) {
            return Message.build(false, "发放定制勋章的用户列表不能为空");
        }

        UserMedalGrantInfoEntity userMedalGrantInfoEntity = convertMedalGrantInfoEntity(param);

        if (Objects.isNull(param.getGrantRecordId())) {
            userMedalGrantInfoEntity.setId(sequenceCreater.nextLongId());
            userMedalGrantInfoEntity.setStatus(MedalGrantStatusEnum.WAIT_CONFIRM.getCode());
            userMedalGrantInfoEntity.setCreateTime(new Date());
            userMedalGrantInfoEntity.setModifyTime(new Date());
            userMedalGrantInfoEntity.setCreator(SecurityUtils.currentUserId());

            //执行新增逻辑
            userMedalGrantInfoMapper.insertSelective(userMedalGrantInfoEntity);
        } else {
            userMedalGrantInfoEntity.setModifyTime(new Date());
            //执行修改逻辑
            userMedalGrantInfoMapper.updateByPrimaryKeySelective(userMedalGrantInfoEntity);
        }

        return Message.build(true);
    }

    @Override
    public Message confirmGrant(ConfirmRecordParam param) {
        UserMedalGrantInfoEntity userMedalGrantInfoEntity = userMedalGrantInfoMapper.selectByPrimaryKey(param.getGrantRecordId());
        if (Objects.isNull(userMedalGrantInfoEntity)) {
            return Message.build(false, "发放记录不存在");
        }
        // 如果不是待确认则不能进行操作
        if (Boolean.FALSE.equals(MedalGrantStatusEnum.WAIT_CONFIRM.getCode().equals(userMedalGrantInfoEntity.getStatus()))) {
            return Message.build(false, "发放状态必须在待确认才能操作");
        }

        UserMedalGrantInfoEntity updateStatusEntity = new UserMedalGrantInfoEntity();

        updateStatusEntity.setId(param.getGrantRecordId());
        updateStatusEntity.setStatus(MedalGrantStatusEnum.WAIT_GRANT.getCode());

        userMedalGrantInfoMapper.updateByPrimaryKeySelective(updateStatusEntity);

        return Message.build(true);
    }

    @Override
    public Message cancelGrant(ConfirmRecordParam param) {
        UserMedalGrantInfoEntity userMedalGrantInfoEntity = userMedalGrantInfoMapper.selectByPrimaryKey(param.getGrantRecordId());
        if (Objects.isNull(userMedalGrantInfoEntity)) {
            return Message.build(false, "发放记录不存在");
        }
        // 如果不是待确认则不能进行操作
        if (Boolean.FALSE.equals(MedalGrantStatusEnum.WAIT_GRANT.getCode().equals(userMedalGrantInfoEntity.getStatus()))) {
            return Message.build(false, "发放状态必须在待发放才能操作");
        }

        UserMedalGrantInfoEntity updateStatusEntity = new UserMedalGrantInfoEntity();

        updateStatusEntity.setId(param.getGrantRecordId());
        updateStatusEntity.setStatus(MedalGrantStatusEnum.WAIT_CONFIRM.getCode());

        userMedalGrantInfoMapper.updateByPrimaryKeySelective(updateStatusEntity);

        return Message.build(true);
    }

    @Override
    public CustomGrantDetailVO getMedalGrantDetail(Long grantRecordId) {
        UserMedalGrantInfoEntity userMedalGrantInfoEntity = userMedalGrantInfoMapper.selectByPrimaryKey(grantRecordId);

        CustomGrantDetailVO customGrantDetailVO = new CustomGrantDetailVO();
        customGrantDetailVO.setGrantRecordId(userMedalGrantInfoEntity.getId());
        customGrantDetailVO.setGrantTime(userMedalGrantInfoEntity.getGrantTime());
        customGrantDetailVO.setMedalId(userMedalGrantInfoEntity.getMedalId());

        String targetUserId = userMedalGrantInfoEntity.getTargetUserId();

        List<Long> grantUserIdList = new ArrayList<>();
        if (StringUtils.isNotBlank(targetUserId)) {
            //数据库中存储的用户id以","分割
            String[] grantUserIds = targetUserId.split(",");
            for (String grantUserId : grantUserIds) {
                grantUserIdList.add(Long.parseLong(grantUserId));
            }
        }

        customGrantDetailVO.setGrantUserIds(grantUserIdList);

        return customGrantDetailVO;
    }

    @Override
    public void grantCustomMedal() {
        try {
            //扫描所有待发放的发放状态并且发放时间早于现在的记录
            List<UserMedalGrantInfoEntity> waitGrantList = userMedalGrantInfoMapper.getGrantListByStatus(MedalGrantStatusEnum.WAIT_GRANT.getCode());

            if (CollectionUtils.isEmpty(waitGrantList)) {
                log.info("无待发放的定制勋章记录");
                return;
            }

            //先都改为发放中
            for (UserMedalGrantInfoEntity userMedalGrantInfoEntity : waitGrantList) {
                UserMedalGrantInfoEntity updateGrantInfo = new UserMedalGrantInfoEntity();
                updateGrantInfo.setId(userMedalGrantInfoEntity.getId());
                updateGrantInfo.setStatus(MedalGrantStatusEnum.GRANTING.getCode());

                userMedalGrantInfoMapper.updateByPrimaryKeySelective(updateGrantInfo);
            }

            //再扫描发放中的记录，进行发放
            List<UserMedalGrantInfoEntity> grantingList = userMedalGrantInfoMapper.getGrantListByStatus(MedalGrantStatusEnum.GRANTING.getCode());

            for (UserMedalGrantInfoEntity userMedalGrantInfoEntity : grantingList) {
                String targetUserId = userMedalGrantInfoEntity.getTargetUserId();
                String[] userIds = targetUserId.split(",");
                for (String userId : userIds) {
                    log.info("发放勋章: {} 给用户: {}", userMedalGrantInfoEntity.getMedalId(), userId);
                    grantCustomMedal(Long.valueOf(userId), userMedalGrantInfoEntity.getMedalId());
                }
            }

            //发放完成，将发放记录改为已发放
            for (UserMedalGrantInfoEntity userMedalGrantInfoEntity : grantingList) {
                UserMedalGrantInfoEntity updateGrantInfo = new UserMedalGrantInfoEntity();
                updateGrantInfo.setId(userMedalGrantInfoEntity.getId());
                updateGrantInfo.setStatus(MedalGrantStatusEnum.ALREADY_GRANT.getCode());

                userMedalGrantInfoMapper.updateByPrimaryKeySelective(updateGrantInfo);
            }
        } catch (Exception e) {
            log.error("勋章发放出现错误", e);
        }

    }

    /**
     * 发放成就勋章
     *
     * @param userId  目标用户id
     * @param medalId 勋章id
     */
    protected void grantCustomMedal(Long userId, Long medalId) {
        UserMedalRelationInfoEntity userMedalRecord = userMedalRelationInfoMapper.getByUserIdAndMedalId(userId, medalId);

        log.info("用户: {} 勋章id: {} 查询出来的勋章信息: {}", userId, medalId, JSON.toJSON(userMedalRecord));

        if (Objects.isNull(userMedalRecord)) {
            log.info("为用户发放成就勋章，用户id：{}, 成就勋章id：{}", userId, medalId);

            UserMedalRelationInfoEntity newGrantInfo = new UserMedalRelationInfoEntity();

            newGrantInfo.setId(sequenceCreater.nextLongId());
            newGrantInfo.setMedalId(medalId);
            newGrantInfo.setCreateTime(new Date());
            newGrantInfo.setModifyTime(new Date());
            newGrantInfo.setHasWear(0);
            newGrantInfo.setWearOrder(0);
            newGrantInfo.setMedalCategory(MedalCategoryEnum.CUSTOM_MEDAL.getCode());
            newGrantInfo.setUserId(userId);

            userMedalRelationInfoMapper.insertSelective(newGrantInfo);

            //增加发放定制勋章个数
            userMedalInfoMapper.increaseGrantNum(medalId);

            //将勋章信息放到用户勋章弹窗缓存中
            UserMedalInfoEntity userMedalInfoEntity = userMedalInfoMapper.selectByPrimaryKey(medalId);
            userMedalPopCacheManager.addUserMedalPopCache(userId, buildPopCacheData(userId, userMedalInfoEntity));
        }
    }

    private UserGrantMedalDTO buildPopCacheData(Long userId, UserMedalInfoEntity userMedalInfoEntity) {
        UserGrantMedalDTO userGrantMedalDTO = new UserGrantMedalDTO();

        userGrantMedalDTO.setMedalId(userMedalInfoEntity.getId());
        userGrantMedalDTO.setGrantDate(new Date());
        userGrantMedalDTO.setMedalImgUrl(userMedalInfoEntity.getGifImgUrl());
        userGrantMedalDTO.setMedalName(userMedalInfoEntity.getName());
        userGrantMedalDTO.setJumpUrl(buildMyMedalPageUrl(userId));

        return userGrantMedalDTO;
    }

    /**
     * 构建跳转我的勋章页面url
     *
     * @param userId 用户id
     * @return 页面地址url
     */
    private String buildMyMedalPageUrl(Long userId) {
        String url = domainIntegrationService.getInnerH5BaseUrl() +
                String.format(h5JumpAddressProperties.getMyMedalUrl(), userId);

        url = UriUtils.encode(url, Charset.defaultCharset());

        return String.format(clientRouteProperties.getWebView(), url, UriUtils.encode("我的勋章", Charset.defaultCharset()));
    }

    /**
     * 转化为勋章发放记录实体类对象
     *
     * @param param 转化对象
     * @return 实现类对象
     */
    private UserMedalGrantInfoEntity convertMedalGrantInfoEntity(SaveCustomMedalGrantParam param) {
        UserMedalGrantInfoEntity userMedalGrantInfoEntity = new UserMedalGrantInfoEntity();

        userMedalGrantInfoEntity.setId(param.getGrantRecordId());
        userMedalGrantInfoEntity.setMedalId(param.getMedalId());
        StringBuilder targetUserIdBuilder = new StringBuilder("");

        for (Long grantUserId : param.getGrantUserIds()) {
            targetUserIdBuilder.append(grantUserId + ",");
        }
        String targetUserId = targetUserIdBuilder.toString();
        if (isNotBlank(targetUserId)) {
            //去掉末尾的
            userMedalGrantInfoEntity.setTargetUserId(targetUserId.substring(0, targetUserId.length() - 1));
        }
        userMedalGrantInfoEntity.setGrantTime(param.getGrantTime());

        return userMedalGrantInfoEntity;
    }
}