package com.bxm.localnews.admin.service.medal.impl;

import com.bxm.localnews.admin.domain.medal.UserMedalInfoMapper;
import com.bxm.localnews.admin.domain.medal.UserMedalRelationInfoMapper;
import com.bxm.localnews.admin.entity.medal.UserMedalInfoEntity;
import com.bxm.localnews.admin.enums.MedalCategoryEnum;
import com.bxm.localnews.admin.enums.MedalStatusEnum;
import com.bxm.localnews.admin.enums.MedalTypeEnum;
import com.bxm.localnews.admin.enums.MedalWearStatusEnum;
import com.bxm.localnews.admin.param.medal.MedalGrantQueryParam;
import com.bxm.localnews.admin.param.medal.MedalQueryParam;
import com.bxm.localnews.admin.param.medal.SaveCustomMedalParam;
import com.bxm.localnews.admin.service.medal.MedalGrantManageService;
import com.bxm.localnews.admin.service.medal.MedalManageService;
import com.bxm.localnews.admin.utils.SecurityUtils;
import com.bxm.localnews.admin.vo.medal.CustomMedalDetailVO;
import com.bxm.localnews.admin.vo.medal.CustomMedalSelectVO;
import com.bxm.localnews.admin.vo.medal.MedalGrantListVO;
import com.bxm.localnews.admin.vo.medal.MedalListVO;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.AllArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 勋章管理服务实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/3/1 4:58 下午
 */
@Service
@AllArgsConstructor
public class MedalManageServiceImpl implements MedalManageService {

    /**
     * 分隔符
     */
    public static final String SEPARATOR = "-";

    private UserMedalInfoMapper userMedalInfoMapper;

    private UserMedalRelationInfoMapper userMedalRelationInfoMapper;

    private SequenceCreater sequenceCreater;

    private MedalGrantManageService medalGrantManageService;

    @Override
    public PageWarper<MedalListVO> getCustomMealList(MedalQueryParam param) {
        return new PageWarper<>(userMedalInfoMapper.getCustomMealList(param));
    }

    @Override
    public Message saveOrUpdateCustomMedal(SaveCustomMedalParam param) {

        if (Objects.isNull(param.getExpiredTime()) || DateUtils.before(param.getExpiredTime(), new Date())) {
            return Message.build(false, "勋章过期时间不能早于现在");
        }

        MedalGrantQueryParam queryParam = new MedalGrantQueryParam();
        queryParam.setMedalId(param.getMedalId());

        PageWarper<MedalGrantListVO> grantList = medalGrantManageService.getGrantList(queryParam);


        if (Objects.nonNull(param.getMedalId())) {
            UserMedalInfoEntity userMedalInfoEntity = userMedalInfoMapper.selectByPrimaryKey(param.getMedalId());
            if (Objects.isNull(userMedalInfoEntity)) {
                return Message.build(false, "勋章不存在");
            }
            if (!CollectionUtils.isEmpty(grantList.getList())) {
                // 已经发放过 判断编辑的字段
                if ((StringUtils.isNotBlank(param.getMedalName())
                        && !Objects.equals(param.getMedalName(), userMedalInfoEntity.getName()))
                        || (StringUtils.isNotBlank(param.getBigImgUrl())
                        && !Objects.equals(param.getBigImgUrl(), userMedalInfoEntity.getActiveImgUrl()))
                        || (StringUtils.isNotBlank(param.getExpiredImgUrl())
                        && !Objects.equals(param.getExpiredImgUrl(), userMedalInfoEntity.getExpiredImgUrl()))
                        || (StringUtils.isNotBlank(param.getThumbnailImgUrl())
                        && !Objects.equals(param.getThumbnailImgUrl(), userMedalInfoEntity.getThumbnailUrl())
                        || (StringUtils.isNotBlank(param.getGifImgUrl())
                        && !Objects.equals(param.getGifImgUrl(), userMedalInfoEntity.getGifImgUrl())))) {

                    return Message.build(false, "勋章发放后，勋章名称、勋章大图、勋章失效图、勋章缩略图不可编辑");
                }
            }
        }

        UserMedalInfoEntity saveMedalEntity = convertCustomMedalEntity(param);
        // 如果为id 为null即保存
        if (Objects.isNull(saveMedalEntity.getId())) {
            saveMedalEntity.setId(sequenceCreater.nextLongId());
            saveMedalEntity.setCategory(MedalCategoryEnum.CUSTOM_MEDAL.getCode());
            saveMedalEntity.setType(MedalTypeEnum.GENERAL_CUSTOM_MEDAL.name());
            saveMedalEntity.setCreateTime(new Date());
            saveMedalEntity.setModifyTime(new Date());
            saveMedalEntity.setCreator(SecurityUtils.currentUserId());

            userMedalInfoMapper.insertSelective(saveMedalEntity);
        } else {
            saveMedalEntity.setModifyTime(new Date());
            //修改有效期后重新上架
            saveMedalEntity.setStatus(MedalStatusEnum.USABLE.getCode());
            //执行更新逻辑
            userMedalInfoMapper.updateByPrimaryKeySelective(saveMedalEntity);
        }
        return Message.build(true);
    }

    @Override
    public List<CustomMedalSelectVO> getUsableCustomMedalList(String keyword, Integer allType) {
        int type = 0;

        try {
            Long.parseLong(keyword);
        } catch (NumberFormatException e) {
            type = 1;
        }
        //如果为空则默认查有效期内的定制勋章
        if (Objects.isNull(allType)) {
            allType = 0;
        }


        List<CustomMedalSelectVO> usableCustomMedalList = userMedalInfoMapper.getUsableCustomMedalList(type, keyword, allType);

        for (CustomMedalSelectVO customMedalSelectVO : usableCustomMedalList) {
            customMedalSelectVO.setMedalDesc(customMedalSelectVO.getMedalDesc() + SEPARATOR + customMedalSelectVO.getMedalId());
        }

        return usableCustomMedalList;
    }

    @Override
    public CustomMedalDetailVO getMedalDetail(Long medalId) {
        UserMedalInfoEntity userMedalInfoEntity = userMedalInfoMapper.selectByPrimaryKey(medalId);

        CustomMedalDetailVO customMedalDetailVO = new CustomMedalDetailVO();
        customMedalDetailVO.setMedalId(userMedalInfoEntity.getId());
        customMedalDetailVO.setBigImgUrl(userMedalInfoEntity.getActiveImgUrl());
        customMedalDetailVO.setExpiredImgUrl(userMedalInfoEntity.getExpiredImgUrl());
        customMedalDetailVO.setThumbnailImgUrl(userMedalInfoEntity.getThumbnailUrl());
        customMedalDetailVO.setGifImgUrl(userMedalInfoEntity.getGifImgUrl());
        customMedalDetailVO.setExpiredTime(userMedalInfoEntity.getExpiredTime());
        customMedalDetailVO.setMedalExplain(userMedalInfoEntity.getExplainText());
        customMedalDetailVO.setMedalName(userMedalInfoEntity.getName());

        return customMedalDetailVO;
    }

    @Override
    public void offlineExpiredCustomMedal() {
        //查出所有的定制勋章处于生效状态的
        List<UserMedalInfoEntity> allUsableMedalList = userMedalInfoMapper.getAllUsableMedal();

        List<Long> expiredMedalIdList = new ArrayList<>();

        //如果过期，则直接进行下架
        for (UserMedalInfoEntity userMedalInfoEntity : allUsableMedalList) {
            if (DateUtils.before(DateUtils.getClearDate(userMedalInfoEntity.getExpiredTime()), DateUtils.getClearDate(new Date()))) {
                expiredMedalIdList.add(userMedalInfoEntity.getId());
            }
        }

        if (CollectionUtils.isNotEmpty(expiredMedalIdList)) {
            userMedalInfoMapper.batchUpdateCustomStatus(expiredMedalIdList, MedalStatusEnum.EXPIRED.getCode());

            //取消佩戴了这个定制勋章的用户
            userMedalRelationInfoMapper.batchUpdateHasWearStatus(expiredMedalIdList, MedalWearStatusEnum.WEAR.getCode());
        }
    }

    /**
     * 转化为定制勋章对象
     *
     * @param param 转化对象
     * @return 实体类对象
     */
    private UserMedalInfoEntity convertCustomMedalEntity(SaveCustomMedalParam param) {
        UserMedalInfoEntity userMedalInfoEntity = new UserMedalInfoEntity();

        userMedalInfoEntity.setId(param.getMedalId());
        userMedalInfoEntity.setName(param.getMedalName());
        userMedalInfoEntity.setExplainText(param.getMedalExplain());
        userMedalInfoEntity.setActiveImgUrl(param.getBigImgUrl());
        userMedalInfoEntity.setThumbnailUrl(param.getThumbnailImgUrl());
        userMedalInfoEntity.setExpiredImgUrl(param.getExpiredImgUrl());
        userMedalInfoEntity.setGifImgUrl(param.getGifImgUrl());
        userMedalInfoEntity.setExpiredTime(param.getExpiredTime());

        return userMedalInfoEntity;
    }
}