package com.bxm.localnews.admin.service.security.impl;

import com.bxm.localnews.admin.common.AdminBizConfigProperties;
import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.UserBlockMapper;
import com.bxm.localnews.admin.dto.UserBlockDTO;
import com.bxm.localnews.admin.enums.BlockUserTypeEnum;
import com.bxm.localnews.admin.integration.IMIntegrationService;
import com.bxm.localnews.admin.integration.PushMsgIntegrationService;
import com.bxm.localnews.admin.integration.UserAuthCodeIntegrationService;
import com.bxm.localnews.admin.param.RemoveAuthCodeParam;
import com.bxm.localnews.admin.param.security.UserAuthCodeParam;
import com.bxm.localnews.admin.service.security.AdminBlockUserService;
import com.bxm.localnews.admin.vo.UserBlock;
import com.bxm.localnews.auth.enums.AuthCodeEnum;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Service
public class AdminBlockUserServiceImpl implements AdminBlockUserService {


    @Autowired
    private UserBlockMapper userBlockMapper;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private IMIntegrationService imIntegrationService;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private UserAuthCodeIntegrationService userAuthCodeIntegrationService;

    @Autowired
    private AdminBizConfigProperties adminBizConfigProperties;

    @Autowired
    private PushMsgIntegrationService pushMsgIntegrationService;

    @Override
    public UserBlockDTO getInfo(Long userId) {
        List<UserBlock> userBlockList = userBlockMapper.selectByUserId(userId);
        if (CollectionUtils.isEmpty(userBlockList)) {
            return null;
        }
        return convert(userId, userBlockList);
    }

    @Override
    public Boolean editUserBlock(UserBlockDTO userBlockDTO) {
        Long userId = userBlockDTO.getUserId();
        Date blockTime = userBlockDTO.getBlockTime();
        List<Integer> index = Lists.newArrayList();
        // 禁止发帖、禁止发小纸条、禁止发评论、禁止私聊、拉黑、推荐降权
        String actions = "";
        //禁止私聊
        if (userBlockDTO.getChatLimitFlag()) {
            limit(userBlockDTO, BlockUserTypeEnum.CHAT_LIMIT, getChatLimitRedisKey(userId));
            imIntegrationService.block(userId, blockTime);
            actions += "禁止私聊、";
        } else {
            cancelLimit(userBlockDTO, BlockUserTypeEnum.CHAT_LIMIT, getChatLimitRedisKey(userId));
            imIntegrationService.unblock(userId);
        }

        //拉黑
        if (userBlockDTO.getBlockFlag()) {
            limit(userBlockDTO, BlockUserTypeEnum.BLOCK_LIMIT, getBlockLimitRedisKey(userId));
            actions += "拉黑、";
        } else {
            cancelLimit(userBlockDTO, BlockUserTypeEnum.BLOCK_LIMIT, getBlockLimitRedisKey(userId));
        }

        //推荐降权
        if (userBlockDTO.getRecommendLimitFlag()) {
            actions += "推荐降权、";
            limit(userBlockDTO, BlockUserTypeEnum.RECOMMEND_LIMIT, getRecommendLimitRedisKey(userId));
        } else {
            cancelLimit(userBlockDTO, BlockUserTypeEnum.RECOMMEND_LIMIT, getRecommendLimitRedisKey(userId));
        }

        //禁止评论
        if (userBlockDTO.getForbidComment()) {
            actions += "禁止评论、";
            this.addForbidUserAuth(AuthCodeEnum.FORBID_COMMENT,userBlockDTO.getUserId());
            limit(userBlockDTO, BlockUserTypeEnum.FORBID_COMMENT, getForbidCommentLimitRedisKey(userId));
            index.add(AuthCodeEnum.FORBID_COMMENT.index);
        } else {
            this.removeForbidUserAuth(AuthCodeEnum.FORBID_COMMENT,userBlockDTO.getUserId());
            this.cancelLimit(userBlockDTO,BlockUserTypeEnum.FORBID_COMMENT,getForbidNoteLimitRedisKey(userId));
        }

        //禁止发帖
        if (userBlockDTO.getForbidPost()) {
            actions += "禁止发帖、";
            this.addForbidUserAuth(AuthCodeEnum.FORBID_POST,userBlockDTO.getUserId());
            limit(userBlockDTO, BlockUserTypeEnum.FORBID_POST, getForbidPostLimitRedisKey(userId));
            index.add(AuthCodeEnum.FORBID_POST.index);
        } else {
            this.removeForbidUserAuth(AuthCodeEnum.FORBID_POST,userBlockDTO.getUserId());
            this.cancelLimit(userBlockDTO,BlockUserTypeEnum.FORBID_POST,getForbidNoteLimitRedisKey(userId));
        }

        //禁止发小纸条
        if (userBlockDTO.getForbidNote()) {
            actions += "禁止发小纸条";
            this.addForbidUserAuth(AuthCodeEnum.FORBID_NOTE,userBlockDTO.getUserId());
            limit(userBlockDTO, BlockUserTypeEnum.FORBID_NOTE, getForbidNoteLimitRedisKey(userId));
            index.add(AuthCodeEnum.FORBID_NOTE.index);
        } else {
            this.removeForbidUserAuth(AuthCodeEnum.FORBID_NOTE,userBlockDTO.getUserId());
            this.cancelLimit(userBlockDTO,BlockUserTypeEnum.FORBID_NOTE,getForbidNoteLimitRedisKey(userId));
        }

        //删除移除权限定时任务
        userAuthCodeIntegrationService.removeAuthTask(userBlockDTO.getUserId());
        if (CollectionUtils.isNotEmpty(index)){
            RemoveAuthCodeParam param = RemoveAuthCodeParam.builder()
                    .authCodeIndex(index)
                    .expireTime(userBlockDTO.getBlockTime())
                    .userId(userBlockDTO.getUserId())
                    .build();
            userAuthCodeIntegrationService.addRemoveAuthTask(param);
        }

        // 发推送
        if (isNotBlank(actions)) {
            if (actions.lastIndexOf(",") == actions.length() - 1) {
                actions = actions.substring(0, actions.length() - 1);
            }
            pushMsgIntegrationService.pushBlockMsg(actions, userBlockDTO.getBlockTime(), userBlockDTO.getUserId(),
                    adminBizConfigProperties.getOfficialRulePostId());
        }

        return Boolean.TRUE;
    }



    private void addForbidUserAuth(AuthCodeEnum ennum, Long userId){
        UserAuthCodeParam param = new UserAuthCodeParam();
        param.setAuthCode(ennum.index);
        param.setUserId(userId);
        userAuthCodeIntegrationService.addAuth(param);
    }

    private void removeForbidUserAuth(AuthCodeEnum ennum, Long userId) {
        UserAuthCodeParam param = new UserAuthCodeParam();
        param.setAuthCode(ennum.index);
        param.setUserId(userId);
        userAuthCodeIntegrationService.delAuth(param);
    }


    private UserBlockDTO convert(Long userId, List<UserBlock> userBlockList) {
        UserBlockDTO userBlockDTO = UserBlockDTO.builder()
                .userId(userId)
                .blockTime(userBlockList.get(0).getBlockTime())
                .build();
        userBlockList.forEach(userBlock -> {
            if (BlockUserTypeEnum.CHAT_LIMIT.getBlockType().equals(userBlock.getBlockType()) &&
                    isNotBlank(redisStringAdapter.getString(getChatLimitRedisKey(userId)))) {
                userBlockDTO.setChatLimitFlag(true);
            }
            if (BlockUserTypeEnum.BLOCK_LIMIT.getBlockType().equals(userBlock.getBlockType()) &&
                    isNotBlank(redisStringAdapter.getString(getBlockLimitRedisKey(userId)))) {
                userBlockDTO.setBlockFlag(true);
            }
            if (BlockUserTypeEnum.RECOMMEND_LIMIT.getBlockType().equals(userBlock.getBlockType()) &&
                    isNotBlank(redisStringAdapter.getString(getRecommendLimitRedisKey(userId)))) {
                userBlockDTO.setRecommendLimitFlag(true);
            }
            if (BlockUserTypeEnum.FORBID_COMMENT.getBlockType().equals(userBlock.getBlockType()) &&
                    isNotBlank(redisStringAdapter.getString(getForbidCommentLimitRedisKey(userId)))) {
                userBlockDTO.setForbidComment(true);
            }
            if (BlockUserTypeEnum.FORBID_POST.getBlockType().equals(userBlock.getBlockType()) &&
                    isNotBlank(redisStringAdapter.getString(getForbidPostLimitRedisKey(userId)))) {
                userBlockDTO.setForbidPost(true);
            }
            if (BlockUserTypeEnum.FORBID_NOTE.getBlockType().equals(userBlock.getBlockType()) &&
                    isNotBlank(redisStringAdapter.getString(getForbidNoteLimitRedisKey(userId))) ) {
                userBlockDTO.setForbidNote(true);
            }
        });

        return userBlockDTO;
    }

    private void cancelLimit(UserBlockDTO userBlockDTO, BlockUserTypeEnum blockUserTypeEnum, KeyGenerator redisKey) {

        Long userId = userBlockDTO.getUserId();
        UserBlock userBlock = new UserBlock();
        userBlock.setUserId(userId);
        userBlock.setBlockType(blockUserTypeEnum.getBlockType());
        userBlockMapper.deleteByUserIdAndType(userBlock);
        if (Objects.nonNull(redisKey)){
            redisStringAdapter.remove(redisKey);
        }
    }

    private void limit(UserBlockDTO userBlockDTO, BlockUserTypeEnum blockUserTypeEnum, KeyGenerator redisKey) {
        Date blockTime = userBlockDTO.getBlockTime();

        addRecord(userBlockDTO, blockUserTypeEnum.getBlockType());

        redisStringAdapter.set(redisKey, blockTime);
        redisStringAdapter.expire(redisKey, blockTime);

    }

    private void addRecord(UserBlockDTO userBlockDTO, Integer blockType) {

        UserBlock userBlock = UserBlock.builder()
                .id(sequenceCreater.nextLongId())
                .userId(userBlockDTO.getUserId())
                .blockTime(userBlockDTO.getBlockTime())
                .blockType(blockType)
                .build();

        userBlockMapper.deleteByUserIdAndType(userBlock);
        userBlockMapper.insertSelective(userBlock);
    }

    private KeyGenerator getChatLimitRedisKey(Long userId) {
        return RedisConfig.IM_BLOCK_LIST.copy().appendKey(userId);
    }

    private KeyGenerator getBlockLimitRedisKey(Long userId) {
        return RedisConfig.USER_BLOCK_LIMIT.copy().appendKey(userId);
    }

    private KeyGenerator getRecommendLimitRedisKey(Long userId) {
        return RedisConfig.RECOMMEND_BLOCK_LIMIT.copy().appendKey(userId);
    }

    private KeyGenerator getForbidCommentLimitRedisKey(Long userId) {
        return RedisConfig.FORBID_COMMENT_LIMIT.copy().appendKey(userId);
    }

    private KeyGenerator getForbidPostLimitRedisKey(Long userId) {
        return RedisConfig.FORBID_POST_LIMIT.copy().appendKey(userId);
    }

    private KeyGenerator getForbidNoteLimitRedisKey(Long userId) {
        return RedisConfig.FORBID_NOTE_LIMIT.copy().appendKey(userId);
    }

}
