package com.bxm.localnews.base.integration;

import com.bxm.localnews.base.facade.DomainFacadeService;
import com.bxm.localnews.base.model.dto.AvailableDomainInfo;
import com.bxm.localnews.base.model.dto.ViewSceneDomain;
import com.bxm.localnews.base.model.param.GetAvailableDomainInfoParam;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.localnews.common.constant.DomainScene.DomainViewScene;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;

import java.util.Objects;
import java.util.Optional;

/**
 * @author gonzo
 * @date 2020-07-30 20:28
 **/
@Slf4j
@Service
@AllArgsConstructor
public class DomainIntegrationService {

    private final DomainFacadeService domainFacadeService;

    /**
     * 根据场景获取域名
     * @param scene 场景值
     * @param appId appId 某些场景如微信环境的授权，是需要appId和域名相互绑定的 所以需要传入app id来获取对应的域名
     * @param viewScene 落地页场景值 如果选择的是站外分享，则需要指定具体的落地页类型 {@link DomainScene.DomainViewScene}
     * @return 包含了域名、baseUrl（如果有需要则会拼接）的信息
     */
    public Optional<AvailableDomainInfo> getBaseUrlInfoByScene(DomainScene scene, DomainScene.DomainViewScene viewScene,
                                                               String appId) {
        try {
            ResponseEntity<AvailableDomainInfo> domainInfo = domainFacadeService.getBaseUrlInfoByScene(Objects.toString(scene),
                    Objects.toString(viewScene), appId);
            if (Objects.nonNull(domainInfo.getBody())) {
                return Optional.of(domainInfo.getBody());
            }
        } catch(Exception e) {
            log.error("根据场景获取域名失败 scene: {} appId: {}", scene, appId, e);
        }

        return Optional.empty();
    }

    /**
     * 根据请求参数，获取具体落地页的域名信息
     * @param viewScene 落地页场景类型
     * @return 落地页的域名信息
     */
    public Optional<ViewSceneDomain> getViewSceneBaseUrl(DomainViewScene viewScene) {
        try {
            ResponseEntity<ViewSceneDomain> domainInfo = domainFacadeService.getViewSceneBaseUrl(Objects.toString(viewScene));
            if (Objects.nonNull(domainInfo.getBody())) {
                return Optional.of(domainInfo.getBody());
            }
        } catch(Exception e) {
            log.error("根据场景获取域名失败 viewScene: {}", viewScene, e);
        }

        return Optional.empty();
    }

    private String getByScene(DomainScene scene) {
        Optional<AvailableDomainInfo> baseUrlInfoByScene = getBaseUrlInfoByScene(scene, null, null);
        if (baseUrlInfoByScene.isPresent()) {
            return baseUrlInfoByScene.get().getBaseUrl();
        }
        return "";
    }

    /**
     * 获取内部H5 base url
     * @return 内部H5 base url
     */
    public String getInnerH5BaseUrl() {
        return getByScene(DomainScene.INNER_H5);
    }

    /**
     * 获取分享的base url
     * @param viewScene 分享落地页的类型
     * @return 分享的base url
     */
    public String getOutSideShareBaseUrl(DomainScene.DomainViewScene viewScene) {
        Optional<AvailableDomainInfo> baseUrlInfoByScene = getBaseUrlInfoByScene(DomainScene.OUTSIDE_SHARE, viewScene, null);
        return baseUrlInfoByScene.isPresent() ? baseUrlInfoByScene.get().getBaseUrl() : "";
    }

    /**
     * 获取服务器的base url
     * @return 服务器的base url
     */
    public String getServerHostBaseUrl() {
        return getByScene(DomainScene.SERVER_HOST);
    }

}
