package com.bxm.localnews.base.service.impl;

import com.bxm.localnews.base.domain.AreaCodeMapper;
import com.bxm.localnews.base.domain.AreaDivisionMapper;
import com.bxm.localnews.base.domain.LocationMapper;
import com.bxm.localnews.base.dto.LocationDetailDTO;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.base.service.LocationService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.dto.AreaDivisionDTO;
import com.bxm.localnews.common.dto.LocationDTO;
import com.bxm.localnews.common.dto.LocationMeta;
import com.bxm.localnews.common.vo.AreaCode;
import com.bxm.localnews.common.vo.AreaDivision;
import com.bxm.localnews.common.vo.Location;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

@Service
public class LocationServiceImpl extends BaseService implements LocationService, LocationFacadeService {

    private static final int AREA_CODE_LENGTH = 12;

    private final AreaCodeMapper areaCodeMapper;
    @Autowired
    private AreaDivisionMapper areaDivisionMapper;

    @Autowired
    private LocationMapper locationMapper;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    public LocationServiceImpl(AreaCodeMapper areaCodeMapper) {
        this.areaCodeMapper = areaCodeMapper;
    }


    @Override
    public AreaCode selectByAreaName(String name, String province) {
        List<AreaCode> areaCodeList = areaCodeMapper.selectByAreaName(province);
        if (areaCodeList != null) {
            for (AreaCode areaCode : areaCodeList) {
                String tail = areaCode.getAreaCode().substring(2, 6);
                if ("0000".equalsIgnoreCase(tail)) {
                    String codeSplit = areaCode.getAreaCode().substring(0, 2);
                    List<AreaCode> areaList = areaCodeMapper.selectByAreaCode(name, codeSplit);
                    if (areaList != null) {
                        return areaList.get(0);
                    }
                }
            }

        }
        return null;
    }

    @Override
    public LocationMeta listLocation() {
        KeyGenerator hotCityKey = RedisConfig.HOT_LOCATION;
        TypeReference<List<LocationDTO>> hotCityTypeReference = new TypeReference<List<LocationDTO>>() {
        };
        List<LocationDTO> hotLocationList = redisStringAdapter.get(hotCityKey, hotCityTypeReference);
        if (hotLocationList == null) {
            hotLocationList = listHotLocation();
            redisStringAdapter.set(hotCityKey, hotLocationList);
        }

        KeyGenerator openedCityKey = RedisConfig.OPENED_LOCATION;
        TypeReference<List<LocationDTO>> openedCityTypeReference = new TypeReference<List<LocationDTO>>() {
        };
        List<LocationDTO> openedLocationList = redisStringAdapter.get(openedCityKey, openedCityTypeReference);
        if (openedLocationList == null) {
            openedLocationList = listOpenedLocation();
            redisStringAdapter.set(openedCityKey, openedLocationList);
        }

        LocationMeta locationMeta = new LocationMeta();
        locationMeta.setHotLocation(hotLocationList);
        locationMeta.setOpenedLocation(openedLocationList);

        return locationMeta;
    }

    @Override
    public List<AreaDivisionDTO> listAreaDivisionDTO(String parentCode) {
        AreaDivision areaDivisionParam = new AreaDivision();
        areaDivisionParam.setParentCode(parentCode);
        List<AreaDivision> areaDivisionList = areaDivisionMapper.selectByModel(areaDivisionParam);
        return areaDivisionList.stream().map(this::convertDivision).collect(Collectors.toList());
    }

    private AreaDivisionDTO convertDivision(AreaDivision areaDivision) {
        AreaDivisionDTO areaDivisionDTO = new AreaDivisionDTO();
        areaDivisionDTO.setCode(areaDivision.getCode());
        areaDivisionDTO.setLevel(areaDivision.getLevel());
        areaDivisionDTO.setName(areaDivision.getName());
        return areaDivisionDTO;
    }

    private List<LocationDTO> listHotLocation() {
        Location locationParam = new Location();
        locationParam.setHot((byte) 1);
        locationParam.setShow((byte) 1);
        List<Location> locationList = locationMapper.selectByModel(locationParam);
        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        return locationDTOList;
    }

    private List<LocationDTO> listOpenedLocation() {
        Location locationParam = new Location();
        locationParam.setShow((byte) 1);
        List<Location> locationList = locationMapper.selectByModel(locationParam);


        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        //填充首字母
        locationDTOList.forEach(x -> x.setPinyin(x.getPinyin().substring(0, 1).toUpperCase()));

        return locationDTOList;
    }

    private LocationDTO convertLocation(Location location) {
        LocationDTO locationDTO = new LocationDTO();
        locationDTO.setCode(location.getCode());
        locationDTO.setHot(location.getHot());
        locationDTO.setLevel(location.getLevel());
        locationDTO.setName(location.getName());
        locationDTO.setPinyin(location.getPinyin().substring(0, 1).toUpperCase());
        return locationDTO;
    }

    @Override
    public com.bxm.localnews.base.dto.LocationDTO getLocationByCode(String code) {
        com.bxm.localnews.base.dto.LocationDTO locationDTO = null;
        Location location = locationMapper.selectByCode(code);
        if (null != location) {
            locationDTO = new com.bxm.localnews.base.dto.LocationDTO();
            BeanUtils.copyProperties(location, locationDTO);
        }

        return locationDTO;
    }

    @Override
    public LocationDetailDTO getLocationDetailByCode(String code) {
        com.bxm.localnews.base.dto.LocationDetailDTO locationDTO = null;
        Location location = locationMapper.selectByCode(code);
        if (null != location) {
            locationDTO = new com.bxm.localnews.base.dto.LocationDetailDTO();
            BeanUtils.copyProperties(location, locationDTO);
        }

        return locationDTO;
    }

    @Override
    public boolean compareLocationCode(String originalCode, List<String> standardCodeList) {
        logger.debug("参加活动地区编码相比较，用户地区编号：{} ，活动地区列表：{}",originalCode,standardCodeList);
        //直接以12位编码查询
        if (null != originalCode && CollectionUtils.isNotEmpty(standardCodeList)) {
            String fullCode = originalCode;
            if (AREA_CODE_LENGTH != fullCode.length()) {
                //当定位时取得的是6位城市编码
                fullCode = originalCode + "000000";
                com.bxm.localnews.base.dto.LocationDTO locationDTO = getLocationByCode(fullCode);
                //如果是区是查不到的，查询上一级
                if (null == locationDTO) {
                    fullCode = originalCode.substring(0, 4) + "00000000";
                }
            }
            if (standardCodeList.contains(fullCode)) {
                return true;
            }
        }
        return false;
    }
}
