package com.bxm.localnews.base.service.impl;

import com.bxm.localnews.base.config.DomainConfig;
import com.bxm.localnews.base.domain.DomainExtendMapper;
import com.bxm.localnews.base.service.DomainFacadeService;
import com.bxm.localnews.base.service.domain.SceneBaseUrlJoiner;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.localnews.common.constant.DomainScene.DomainViewScene;
import com.bxm.localnews.common.entity.DomainInfo;
import com.bxm.localnews.common.param.GetAvailableDomainInfoParam;
import com.bxm.localnews.common.param.GetViewSceneDomainInfoParam;
import com.bxm.localnews.common.param.SelectAvailableDomainBySceneParam;
import com.bxm.localnews.common.vo.AvailableDomainInfo;
import com.bxm.localnews.common.vo.ViewSceneDomain;
import com.gexin.fastjson.JSON;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * @author Gonzo
 * @date 2020-07-30 17:21
 */
@Slf4j
@Service
public class DomainServiceImpl implements DomainService, DomainFacadeService {

    private final DomainExtendMapper domainExtendMapper;

    private final SceneBaseUrlJoiner sceneBaseUrlJoiner;

    private final DomainConfig domainConfig;

    private LoadingCache<SelectAvailableDomainBySceneParam, List<DomainInfo>> CACHE;

    public DomainServiceImpl(DomainExtendMapper domainExtendMapper, SceneBaseUrlJoiner sceneBaseUrlJoiner,
                             DomainConfig domainConfig) {
        this.domainExtendMapper = domainExtendMapper;
        this.sceneBaseUrlJoiner = sceneBaseUrlJoiner;
        this.domainConfig = domainConfig;
    }

    @Override
    public Optional<AvailableDomainInfo> getDomainByScene(GetAvailableDomainInfoParam param) {
        Optional<DomainScene> domainScene = DomainScene.forName(param.getScene());
        // 如果没有场景 则返回空
        if (!domainScene.isPresent()) {
            log.warn("请求参数: {} 的场景值不存在 无法获取域名信息", JSON.toJSONString(param));
            return Optional.empty();
        }

        // 获取落地页场景
        Optional<DomainViewScene> domainViewScene = DomainViewScene.forName(param.getViewScene());
        if (Objects.equals(domainScene.get(), DomainScene.OUTSIDE_SHARE_VIEW) && !domainViewScene.isPresent()) {
            log.warn("请求参数: {} 的落地页场景值不存在 无法获取域名信息", JSON.toJSONString(param));
            return Optional.empty();
        }

        // 根据场景类型 查询可用的域名 这里不需要穿落地页类型
        List<DomainInfo> domainInfos = selectAvailableDomainBySceneByCache(domainScene.get().getScene(),
                null,
                param.getAppId());
        if (CollectionUtils.isEmpty(domainInfos)) {
            return Optional.empty();
        }

        // 获取第一个
        DomainInfo domainInfo = domainInfos.get(0);

        AvailableDomainInfo info = new AvailableDomainInfo();
        info.setDomain(domainInfo.getDomain());
        // 拼接
        info.setBaseUrl(sceneBaseUrlJoiner.joinByScene(domainScene.get(), domainViewScene.orElse(null),
                StringUtils.join(domainInfo.getProtocol(),
                domainInfo.getDomain())));
        return Optional.of(info);
    }


    @Override
    public AvailableDomainInfo getDomainInfo(GetAvailableDomainInfoParam param) {
        return getDomainByScene(param).orElse(new AvailableDomainInfo());
    }

    @Override
    public ViewSceneDomain getViewSceneDomain(GetViewSceneDomainInfoParam param) {
        Optional<DomainViewScene> domainScene = DomainViewScene.forName(param.getViewScene());

        // 如果没有场景 则返回空
        if (!domainScene.isPresent()) {
            return new ViewSceneDomain();
        }

        // 根据落地页场景 获取对应的落地页域名
        List<DomainInfo> domainInfos = selectAvailableDomainBySceneByCache(DomainScene.OUTSIDE_SHARE_VIEW.getScene(),
                domainScene.get().getScene(), null);
        if (CollectionUtils.isEmpty(domainInfos)) {
            return new ViewSceneDomain();
        }
        // 获取第一个
        DomainInfo domainInfo = domainInfos.get(0);

        ViewSceneDomain domain = new ViewSceneDomain();
        domain.setDomain(domainInfo.getDomain());
        domain.setBaseUrl(sceneBaseUrlJoiner.joinByScene(DomainScene.OUTSIDE_SHARE_VIEW, null,
                StringUtils.join(domainInfo.getProtocol(),
                domainInfo.getDomain())));
        domain.setViewScene(Objects.toString(domainScene));
        return domain;
    }

    /**
     * 根据场景获取可用的域名信息列表 优先查询缓存
     *
     * @param scene     required 场景值
     * @param viewScene 落地页场景
     * @param appId     option 某些场景是在微信环境下的，所以选用appid对应的域名
     * @return 域名列表
     */
    private List<DomainInfo> selectAvailableDomainBySceneByCache(byte scene, Byte viewScene, String appId) {

        SelectAvailableDomainBySceneParam param = new SelectAvailableDomainBySceneParam();
        param.setScene(scene);
        param.setViewScene(viewScene);
        param.setAppId(appId);
        return selectAvailableDomainBySceneByCache(param);
    }

    private List<DomainInfo> selectAvailableDomainBySceneByCache(SelectAvailableDomainBySceneParam param) {

        if (Objects.isNull(CACHE)) {
            CACHE = CacheBuilder.newBuilder()
                    .maximumSize(100)
                    // 1h失效
                    .expireAfterWrite(1, TimeUnit.HOURS)
                    .build(new CacheLoader<SelectAvailableDomainBySceneParam, List<DomainInfo>>() {
                        @Override
                        public List<DomainInfo> load(SelectAvailableDomainBySceneParam sceneParam) throws Exception {
                            // 根据落地页场景 获取对应的落地页域名
                            return domainExtendMapper.selectAvailableDomainByScene(sceneParam.getScene(),
                                    sceneParam.getViewScene(), sceneParam.getAppId());
                        }
                    });
        }

        // 如果缓存开关未开启 则每次都刷新一下 请求最新的
        if (!Objects.equals(domainConfig.getCacheSwitch(), Boolean.TRUE)) {
            CACHE.refresh(param);
        }

        return CACHE.getUnchecked(param);
    }
}
