package com.bxm.localnews.market.integration;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.market.dto.CashCouponDTO;
import com.bxm.localnews.market.facade.MerchantCouponFeignService;
import com.bxm.localnews.market.param.UserCouponOrderParam;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Objects;

/**
 * @author liujia
 * @date 2020-09-28 14:36
 **/
@Slf4j
@Service
public class MerchantCouponIntegrationService {

    @Resource
    private MerchantCouponFeignService merchantCouponFeignService;

    /**
     * 获取用户优惠券的相关信息，如果优惠券不可用或不存在则返回null
     *
     * @param userCouponId 用户优惠券ID
     * @return 优惠券信息，可能为null
     */
    public CashCouponDTO getUserCouponInfo(Long userCouponId) {
        try {
            ResponseEntity<CashCouponDTO> responseEntity = merchantCouponFeignService.getUserCouponInfo(userCouponId);
            if (null != responseEntity && responseEntity.hasBody()) {
                return responseEntity.getBody();
            }
        } catch (Exception e) {
            log.error("获取用户优惠券失败 userCouponId: {}", userCouponId, e);
        }


        return null;
    }

    /**
     * 订单创建时锁定优惠券，防止多次使用
     *
     * @param param 请求参数
     * @return 锁定结果
     */
    public Message lockUserCoupon(UserCouponOrderParam param) {
        try {
            ResponseEntity<Message> responseEntity = merchantCouponFeignService.lockUserCoupon(param);
            if (responseEntity.hasBody()) {
                return responseEntity.getBody();
            }
        } catch (Exception e) {
            log.error("锁定优惠券失败, param: {}", JSON.toJSONString(param), e);
        }

        return Message.build(false);
    }

    /**
     * 订单失效或取消时，释放优惠券信息，回复正常使用状态
     *
     * @param param 请求参数
     * @return 释放结果
     */
    public Message releaseUserCoupon(UserCouponOrderParam param) {
        try {
            ResponseEntity<Message> responseEntity = merchantCouponFeignService.releaseUserCoupon(param);
            if (Objects.nonNull(responseEntity) && responseEntity.hasBody()) {
                return responseEntity.getBody();
            }
        } catch (Exception e) {
            log.error("释放优惠券失败, param: {}", JSON.toJSONString(param), e);
        }

        return Message.build(false);
    }

    /**
     * 用户支付成功后，核销优惠券
     *
     * @param param 请求参数
     * @return 核销参数
     */
    public Message consumeUserCoupon(UserCouponOrderParam param) {
        try {
            ResponseEntity<Message> responseEntity = merchantCouponFeignService.consumeUserCoupon(param);
            if (responseEntity.hasBody()) {
                return responseEntity.getBody();
            }
        } catch (Exception e) {
            log.error("核销优惠券失败, param: {}", JSON.toJSONString(param), e);
        }


        return Message.build(false);
    }
}
