package com.bxm.localnews.market.integration;

import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.market.dto.UserInfoDTO;
import com.bxm.localnews.market.model.entity.CommissionOrderInfo;
import com.bxm.localnews.market.param.VipQueryParam;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.PushSoundEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.msg.sender.MessageSender;
import com.bxm.localnews.user.model.VipInfoDetailDTO;
import com.bxm.newidea.component.service.BaseService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.web.util.UriUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.nio.charset.Charset;

/**
 * 消息推送服务
 *
 * @author wzy
 * @date 2020/8/6 18:29
 **/
@Service
@Slf4j
public class PushIntegrationService extends BaseService {

    @Resource
    private DomainIntegrationService domainIntegrationService;

    @Resource
    private UserVipIntegrationService userVipIntegrationService;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private MessageSender messageSender;

    /**
     * 推送省钱消息
     *
     * @param orderInfo  订单信息
     * @param saveAmount 省钱金额
     */
    public void pushSaveMoneyMsg(CommissionOrderInfo orderInfo, BigDecimal saveAmount) {
        UserInfoDTO userInfoDTO = userIntegrationService.selectUserFromCache(orderInfo.getUserId());
        if (userInfoDTO == null) {
            log.error("发送省钱推送，获取用户信息失败，userId:{}", orderInfo.getUserId());
            return;
        }

        String locationCode = userInfoDTO.getLocationCode();

        VipInfoDetailDTO vipDetailDTO = userVipIntegrationService.getVipDetail(buildVipQueryParam(orderInfo, locationCode));
        if (vipDetailDTO == null) {
            log.error("发送省钱推送，获取用户vip详细信息失败，userId:{}", orderInfo.getUserId());
            return;
        }

        String title = "省钱提示";
        String content = String.format("你的订单“%s”跟单成功，为您节省了%s元，领先%s的%s人",
                orderInfo.getGoodsName(),
                saveAmount.toString(),
                vipDetailDTO.getRebatePercent(),
                userInfoDTO.getLocationShortName());

        PushMessage pushMessage = getPushMessage(orderInfo, title, content);
        messageSender.sendPushMessage(pushMessage);
    }

    /**
     * 构建vip详细信息查询参数
     *
     * @param orderInfo 订单信息对象
     * @return vip详细信息查询参数
     */
    private VipQueryParam buildVipQueryParam(CommissionOrderInfo orderInfo, String locationCode) {
        VipQueryParam queryParam = new VipQueryParam();
        String scene = "MAIN";
        queryParam.setScene(scene);
        queryParam.setUserId(orderInfo.getUserId());
        queryParam.setAreaCode(locationCode);
        return queryParam;
    }

    /**
     * 获取推送信息对象
     *
     * @param orderInfo 订单信息
     * @param title     推送标题
     * @param content   推送内容
     * @return 推送信息对象
     */
    private PushMessage getPushMessage(CommissionOrderInfo orderInfo, String title, String content) {
        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.JUMP_TYPE);
        info.setProtocol(buildProtocol(orderInfo));

        PushMessage message = PushMessage.build();
        message.setTitle(title);
        message.setContent(content);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPayloadInfo(info);
        message.setSound(PushSoundEnum.GOLD_SOUND);

        message.assign(orderInfo.getUserId());
        return message;
    }

    /**
     * 构建protocol
     *
     * @param orderInfo 订单信息对象
     * @return protocol地址
     */
    private String buildProtocol(CommissionOrderInfo orderInfo) {
        String url = domainIntegrationService.getInnerH5BaseUrl() +
                "tk/member.html#/app/vip?deviceNo={deviceNo}" +
                "&userId=" + orderInfo.getUserId() +
                "&areaCode={areaCode}" +
                "&areaName={areaName}";

        url = UriUtils.encode(url, Charset.defaultCharset());

        return "wst://function/tabbarIndex?className_iOS=TTNormalWebVC&tabName=vip&url=" + url;
    }
}
