package com.bxm.localnews.market.integration;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.market.dto.*;
import com.bxm.localnews.market.facade.UserFeignService;
import com.bxm.localnews.market.param.IncreaseTalentDataParam;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

@Slf4j
@Service
public class UserIntegrationService {

    @Resource
    private UserFeignService userFeignService;

    public UserInfoDTO selectUserFromCache(Long userId) {
        try {
            ResponseEntity<UserInfoDTO> userInfoDTOResponseEntity = this.userFeignService.selectUserFromCache(userId);
            if (Objects.nonNull(userInfoDTOResponseEntity) && userInfoDTOResponseEntity.hasBody()) {
                return userInfoDTOResponseEntity.getBody();
            }
        } catch (Exception e) {
            log.error("获取用户信息失败userId: {}", userId, e);
        }
        return null;
    }

    /**
     * 根据渠道id获取用户信息
     *
     * @param relationId 渠道id
     * @return 用户信息
     */
    public UserTbkInfo selectUserFromCacheByRelationId(String relationId) {

        try {
            ResponseEntity<UserTbkInfo> userTbkInfoResponseEntity = userFeignService.selectUserFromCacheByRelationId(relationId);
            if (Objects.nonNull(userTbkInfoResponseEntity)) {
                return userTbkInfoResponseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("获取用户信息失败relationId :{}", relationId, e);
        }
        return null;
    }

    /**
     * 根据 userId
     *
     * @param userId userId
     * @return 用户信息
     */
    public UserTbkInfo selectUserFromCacheByUserId(Long userId) {

        try {
            ResponseEntity<UserTbkInfo> userTbkInfoResponseEntity = userFeignService.selectUserFromCacheByUserId(userId);
            if (Objects.nonNull(userTbkInfoResponseEntity)) {
                return userTbkInfoResponseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("获取用户信息失败 userId :{}", userId, e);
        }
        return null;
    }

    /**
     * 更新标志位
     *
     * @param userId
     * @param judgeMarker
     * @return
     */
    public Boolean updateUserJudgeMarker(Long userId, LocalNewsUserJudgeMarkerEnum judgeMarker, boolean add) {

        if (log.isDebugEnabled()) {
            log.debug("userId: {} 更新标志位judgeMarker: {} ", userId, judgeMarker);
        }

        try {
            ResponseEntity<Boolean> responseEntity = userFeignService.updateUserJudgeMarker(userId, judgeMarker, add);

            if (Objects.nonNull(responseEntity)) {
                return responseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("更新标志位失败 userId: {} judgeMarker: {}", userId, judgeMarker, e);
        }

        return false;
    }


    /**
     * 根据脱敏手机号码模糊匹配用户信息
     *
     * @param sensitivePhone 脱敏手机号码
     * @return 用户手机号码信息
     */
    public List<UserPhoneDTO> getUserInfoBySensitivePhone(String sensitivePhone) {

        try {
            ResponseEntity<List<UserPhoneDTO>> userPhoneInfoResponseEntity = userFeignService.getUserInfoBySensitivePhone(sensitivePhone);
            if (Objects.nonNull(userPhoneInfoResponseEntity)) {
                return userPhoneInfoResponseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("根据用户脱敏手机号码，获取用户手机号码信息失败 sensitivePhone :{}", sensitivePhone, e);
        }
        return Collections.emptyList();
    }

    /**
     * 根据用户手机号码获取用户id
     *
     * @param phone 手机号码
     * @return id信息
     */
    public UserPhoneDTO getUserInfoByPhone(String phone) {
        try {
            ResponseEntity<UserPhoneDTO> userPhoneInfoResponseEntity = userFeignService.getUserInfoByPhone(phone);
            if (Objects.nonNull(userPhoneInfoResponseEntity)) {
                return userPhoneInfoResponseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("根据用户手机号码，获取用户id信息失败 phone :{}", phone, e);
        }
        return null;
    }

    /***
     * 获取邀请好友的信息
     * @param userId ： 用户id
     * @return ： 返回邀请记录
     */
    public UserInviteHistoryDTO getInviteInfo(Long userId) {
        ResponseEntity<UserInviteHistoryDTO> userPhoneInfoResponseEntity = userFeignService.getInviteInfo(userId);
        if (Objects.nonNull(userPhoneInfoResponseEntity)) {
            UserInviteHistoryDTO inviteHistoryDTO = userPhoneInfoResponseEntity.getBody();
            if (null == inviteHistoryDTO || Objects.equals(inviteHistoryDTO.getState(), AppConst.USER_STATE.NORMAL)) {
                return inviteHistoryDTO;
            }
        }
        return null;
    }

    /**
     * 增加达人贡献订单数量
     *
     * @param param param
     * @return 操作结果
     */
    public Message increaseTalentData(IncreaseTalentDataParam param) {
        try {
            ResponseEntity<Message> messageResponseEntity = userFeignService.increaseTalentData(param);

            if (Objects.nonNull(messageResponseEntity) && messageResponseEntity.hasBody()) {
                return messageResponseEntity.getBody();
            }
        } catch (Exception e) {
            log.error("增加达人贡献订单数量失败, param: {}", JSON.toJSONString(param), e);
        }

        return Message.build(false);
    }

    /**
     * 获取暂时绑定关系接口
     *
     * @param userId userId
     * @return 暂时绑定关系接口
     */
    public TalentBindDTO getBindInfo(Long userId) {
        try {
            ResponseEntity<TalentBindDTO> bindInfo = userFeignService.getBindInfo(userId);

            if (Objects.nonNull(bindInfo) && bindInfo.hasBody()) {
                return bindInfo.getBody();
            }
        } catch (Exception e) {
            log.error("获取暂时绑定关系失败, userId: {}", userId, e);
        }

        return null;
    }


    /**
     * 获取达人的上级和上上级达人信息
     *
     * @param userId userId
     * @return 团队信息
     */
    public TalentSuperiorInfoDTO getSuperiorInfo(Long userId) {
        try {
            ResponseEntity<TalentSuperiorInfoDTO> superiorInfo = userFeignService.getSuperiorInfo(userId);
            if (Objects.nonNull(superiorInfo) && superiorInfo.hasBody()) {
                return superiorInfo.getBody();
            }
        } catch (Exception e) {
            log.error("获取达人的上级和上上级达人信息失败, userId: {}", userId, e);
        }

        return null;
    }

    /**
     * 根据userId获取是否关注微信公众号
     *
     * @param userId userId
     * @return true 已关注
     */
    public boolean isFocusOfficialAccount(Long userId) {
        try {
            ResponseEntity<Boolean> superiorInfo = userFeignService.isFocusOfficialAccount(userId);
            if (Objects.nonNull(superiorInfo) && superiorInfo.hasBody()) {
                return Objects.equals(superiorInfo.getBody(), Boolean.TRUE);
            }
        } catch (Exception e) {
            log.error("获取达人的上级和上上级达人信息失败, userId: {}", userId, e);
        }

        return false;
    }

    /**
     * 判断两个用户之间是否相互关注
     *
     * @param formUserId 关注人
     * @param toUserId   被关注人
     * @return true表示已经关注，或相互关注
     */
    public Boolean isFollow(Long formUserId, Long toUserId) {

        if (Objects.isNull(formUserId) || Objects.isNull(toUserId)) {
            log.warn("参数错误，无法判断是否关注, formUserId: {} toUserId: {} ", formUserId, toUserId);
            return false;
        }

        try {
            ResponseEntity<Boolean> responseEntity = userFeignService.isFollow(formUserId, toUserId);
            if (Objects.nonNull(responseEntity) && responseEntity.hasBody()) {
                return responseEntity.getBody();
            }

        } catch(Exception e) {
            log.error("请求用户服务失败, formUserId: {} toUserId: {} ", formUserId, toUserId, e);
        }
        return false;
    }
}
