package com.bxm.localnews.merchant.controller;

import com.bxm.localnews.merchant.dto.goods.*;
import com.bxm.localnews.merchant.param.goods.*;
import com.bxm.localnews.merchant.service.goods.GoodsBarrageService;
import com.bxm.localnews.merchant.service.goods.GoodsService;
import com.bxm.localnews.merchant.vo.goods.MerchantGoodsListVo;
import com.bxm.localnews.merchants.dto.MerchantInfoDto;
import com.bxm.newidea.component.annotations.ApiVersion;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.bxm.newidea.component.vo.ResponseJson;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author: zhoujincheng
 * @create: 2020-08-20 16:13
 */

@Api(tags = " 11-08 商品相关API")
@RestController
@RequestMapping("{version}/merchant/goods")
public class GoodsController {

    private final GoodsService goodsService;

    private final GoodsBarrageService goodsBarrageService;

    @Autowired
    public GoodsController(GoodsService goodsService, GoodsBarrageService goodsBarrageService) {
        this.goodsService = goodsService;
        this.goodsBarrageService = goodsBarrageService;
    }

    @ApiOperation(value = "11-08-1 【v1】创建商品", notes = "创建商品", httpMethod = "POST")
    @PostMapping("/security/create")
    @ApiVersion(1)
    public ResponseJson<Boolean> create(@RequestBody @Validated GoodsCreateParam goodsCreateParam) {
        Message message = goodsService.create(goodsCreateParam);
        if (message.isSuccess()) {
            ResponseJson.ok(message.getLastMessage());
        }
        return ResponseJson.build(message);
    }

    @ApiOperation(value = "11-08-2 【v1】查看商品详情", notes = "查看商品详情", httpMethod = "GET")
    @GetMapping("/query")
    @ApiVersion(1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "goodsId", value = "商品id", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", required = true)
    })
    public ResponseJson<QueryGoodsDTO> query(@Validated QueryGoodsParam param) {
        Message message = goodsService.query(param);
        if (message.isSuccess()) {
            return ResponseJson.ok(message.getParam("queryGoodsDTO"));
        }
        return ResponseJson.build(message);
    }

    @ApiOperation(value = "11-08-3 【v1】修改商品", notes = "修改商品", httpMethod = "POST")
    @PostMapping("/security/update")
    @ApiVersion(1)
    public ResponseJson<Boolean> update(@RequestBody @Validated GoodsUpdateParam goodsUpdateParam) {
        Message message = goodsService.update(goodsUpdateParam);
        if (message.isSuccess()) {
            ResponseJson.ok(message.getLastMessage());
        }
        return ResponseJson.build(message);
    }

    @ApiOperation(value = "11-08-4 【v1】商品分类下商品列表", notes = "商品分类下商品列表", httpMethod = "GET")
    @GetMapping("/queryAllGoods")
    @ApiVersion(1)
    public ResponseJson<PageWarper<MerchantGoodsListVo>> queryGoodsListAll(QueryAllGoodsParam queryAllGoodsParam) {
        return ResponseJson.ok(goodsService.queryGoodsListAll(queryAllGoodsParam));
    }

    @ApiOperation(value = "11-08-5 【v1】微官网商品列表", notes = "微官网商品列表", httpMethod = "GET")
    @GetMapping("/queryGoodsList")
    @ApiVersion(1)
    public ResponseJson<List<GoodsListDTO>> queryGoodsList(@RequestParam("merchantId") Long merchantId) {
        return ResponseJson.ok(goodsService.queryGoodsList(merchantId).getParam("goodsListDTOList"));
    }

    @ApiOperation(value = "11-08-6 【v1】商品详情页弹幕", notes = "商品详情页头部弹幕", httpMethod = "GET")
    @GetMapping("/barrageTopList")
    @ApiVersion(1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "goodsId", value = "商品id", required = true),
    })
    public ResponseJson<BarrageDTO> barrageTopList(@RequestParam("goodsId") Long goodsId) {
        return ResponseJson.ok(goodsBarrageService.getBarrage(goodsId));
    }


    @ApiOperation(value = "11-08-7 【v1】 撤销或提交审核", notes = "撤销或提交审核", httpMethod = "POST")
    @PostMapping("/security/auditByUser")
    @ApiVersion(1)
    public ResponseJson<Boolean> auditByUser(@RequestBody @Validated AuditByUserParam auditByUserParam) {

        Message message = goodsService.auditByUser(auditByUserParam);
        if (message.isSuccess()) {
            ResponseJson.ok(message.getLastMessage());
        }
        return ResponseJson.build(message);
    }

    @ApiOperation(value = "11-08-8 【v1】 上架或下架", notes = "上架或下架", httpMethod = "POST")
    @PostMapping("/security/shelf")
    @ApiVersion(1)
    public ResponseJson<Boolean> shelf(@RequestBody @Validated ShelfByUserParam shelfByUserParam) {
        Message message = goodsService.shelf(shelfByUserParam);
        if (message.isSuccess()) {
            ResponseJson.ok(message.getLastMessage());
        }
        return ResponseJson.build(message);
    }

    @ApiOperation(value = "11-08-9 【v1】 删除商品", notes = "删除商品", httpMethod = "POST")
    @PostMapping("/security/delete")
    @ApiVersion(1)
    public ResponseJson<Boolean> delete(@RequestBody @Validated GoodsIdParam goodsIdParam) {
        Message message = goodsService.delete(goodsIdParam);
        if (message.isSuccess()) {
            ResponseJson.ok(message.getLastMessage());
        }
        return ResponseJson.build(message);
    }

    @ApiOperation(value = "11-08-10 【v1】 示例商品", notes = "示例商品", httpMethod = "GET")
    @GetMapping("/queryExampleGoods")
    @ApiVersion(1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "merchantId", value = "商户id", required = true),
    })
    public ResponseJson<QueryGoodsDTO> queryExampleGoods(@RequestParam("merchantId") Long merchantId) {
        Message message = goodsService.queryExampleGoods(merchantId);
        if (message.isSuccess()) {
            return ResponseJson.ok(message.getParam("exampleGoodsDTO"));
        }
        return ResponseJson.build(message);
    }


    @ApiOperation(value = "11-08-11 【v1】 商品收藏或移除", notes = "商品收藏或移除", httpMethod = "POST")
    @PostMapping("/security/collection")
    @ApiVersion(1)
    public ResponseJson collection(@RequestBody @Validated GoodsLikeParam goodsLikeParam) {
        Message message = goodsService.collection(goodsLikeParam);
        if (message.isSuccess()) {
            return ResponseJson.ok(Boolean.TRUE);
        }
        return ResponseJson.build(message);
    }

    @GetMapping("/collectionList")
    @ApiOperation(value = "11-08-12 [v1]用户商品收藏列表", notes = "用户商品收藏列表")
    @ApiVersion(1)
    public ResponseJson<PageWarper<CollectionListDTO>> collectionList(@Validated CollectionListParam collectionListParam) {
        return ResponseJson.ok(goodsService.collectionList(collectionListParam));
    }

    @ApiOperation(value = "11-08-13 【v1】查询商品分类下的数量", notes = "查询商品分类下的数量", httpMethod = "GET")
    @GetMapping("/queryAllGoodsSum")
    @ApiVersion(1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "merchantId", value = "商家id", required = true),
    })
    public ResponseJson<AllGoodsSumDTO> queryAllGoodsSum(@RequestParam("merchantId") Long merchantId) {
        Message message = goodsService.queryAllGoodsSum(merchantId);
        if (message.isSuccess()) {
            return ResponseJson.ok(message.getParam("allGoodsSumDTO"));
        }
        return ResponseJson.build(message);
    }

    @ApiOperation(value = "11-08-14 【v1】商品排序 ", notes = "商品排序", httpMethod = "POST")
    @PostMapping("/security/sort")
    @ApiVersion(1)
    public ResponseJson<Boolean> sort(@RequestBody @Validated SortParam sortParam) {
        Message message = goodsService.sort(sortParam);
        if (message.isSuccess()) {
            ResponseJson.ok(message.getLastMessage());
        }
        return ResponseJson.build(message);
    }

    @ApiOperation(value = "11-08-15 【v1】店铺详情", notes = "店铺详情", httpMethod = "GET")
    @GetMapping("/getMerchantInfo")
    @ApiVersion(1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "merchantId", value = "商户id", required = true),
    })
    public ResponseJson<MerchantInfoDto> getMerchantInfo(@RequestParam("merchantId") Long merchantId) {
        Message message = goodsService.getMerchantInfo(merchantId);
        if (message.isSuccess()) {
            return ResponseJson.ok(message.getParam("merchantDto"));
        }
        return ResponseJson.build(message);
    }
}















