package com.bxm.localnews.merchant.integration;


import com.bxm.localnews.merchant.dto.LocationDTO;
import com.bxm.localnews.merchant.facade.LocationFeignService;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 地区信息获取集成机务
 *
 * @author pf.w
 * @date 2020/7/13 20:07
 */
@Slf4j
@Service
public class LocationIntegrationService {

    @Resource
    private LocationFeignService locationFeignService;

    private LoadingCache<String, LocationDTO> cache;

    public LocationDTO getLocationByGeocode(String code) {
        if (StringUtils.isBlank(code)) {
            return new LocationDTO();
        }

        if (cache == null) {
            initCache();
        }
        return cache.getUnchecked(code);
    }

    private void initCache() {
        cache = CacheBuilder.newBuilder()
                .expireAfterWrite(3, TimeUnit.MINUTES)
                .maximumSize(1000)
                .build(new CacheLoader<String, LocationDTO>() {
                    @Override
                    public LocationDTO load(String key) {
                        LocationDTO body = null;
                        try {
                            ResponseEntity<LocationDTO> responseEntity = locationFeignService.getLocationByGeocode(key);
                            if (Objects.nonNull(responseEntity)) {
                                body = responseEntity.getBody();
                            }
                        } catch(Exception e) {
                            log.error("请求biz服务获取城市信息失败, key: {} ", key, e);
                        }


                        if (body == null) {
                            body = new LocationDTO();
                            body.setCode(key);
                        }
                        return body;
                    }
                });
    }
}
