package com.bxm.localnews.msg.platform.impl;

import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.config.AppPushProperties;
import com.bxm.localnews.msg.push.PushResponseCallback;
import com.bxm.localnews.msg.push.Response;
import com.bxm.localnews.msg.vo.MsgUserTokenBean;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Lists;
import com.huawei.push.javasdk.exception.HuaweiMesssagingException;
import com.huawei.push.javasdk.message.*;
import com.huawei.push.javasdk.messaging.*;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;

/**
 * 华为推送平台
 *
 * @author liujia
 */
@Component
@RefreshScope
public class HuaweiPushPlatform extends AbstractPushPlatform implements ApplicationRunner {
    @Resource
    private AppPushProperties appPushProperties;

    private HuaweiApp app;

    private static final int SUCCESS_CODE = 80000000;

    private static final int TOKEN_EXPIRED = 907122032;

    @Override
    public void push(PushMessage message, MsgUserTokenBean userToken, PushResponseCallback callback) {
        Notification notification = new Notification(message.getTitle(), message.getContent());

        ClickAction.Builder clickActionBuilder = ClickAction.builder().setType(1);

        if (message.getActionType() != null) {
            String intent = "intent://" + appPushProperties.getAppPkgName() + "/push?message=" + message.getPayloadInfo().toJsonString() +
                    "#Intent;scheme=route;launchFlags=0x20000;end";
            clickActionBuilder.setIntent(intent);
        }

        try {
            AndroidNotification androidNotification = AndroidNotification.builder()
                    .setTitle(message.getTitle())
                    .setBody(message.getContent())
                    .setClickAction(clickActionBuilder.build())
                    .build();

            AndroidConfig androidConfig = AndroidConfig.builder()
                    .setNotification(androidNotification)
                    .setBiTag("Trump")
                    .build();

            Message huaweiMsg = Message.builder()
                    .setNotification(notification)
                    .setAndroidConfig(androidConfig)
                    .addAllToken(Lists.newArrayList(userToken.getToken()))
                    .build();

            SendResponse response = HuaweiMessaging.getInstance(app).send(huaweiMsg, false);

            if (SUCCESS_CODE == response.getCode()) {
                callback.apply(message, userToken, Response.success());
            } else if (TOKEN_EXPIRED == response.getCode()) {
                logger.info("用户的华为推送令牌已失效：{}", userToken.getUserId());
                callback.apply(message, userToken, Response.expire(response.getMsg()));
            } else {
                logger.info("华为推送失败，失败原因：{},消息内容：{},推送目标：{}", response.getMsg(), message, userToken);
                callback.apply(message, userToken, Response.fail(response.getMsg()));
            }
        } catch (HuaweiMesssagingException e) {
            if (StringUtils.equals("80300007", e.getErrorCode())) {
                callback.apply(message, userToken, Response.expire(e.getMessage()));
            } else {
                callback.apply(message, userToken, Response.fail(e.getMessage()));
            }
            logger.error(e.getMessage(), e);
        }
    }

    @Override
    public PlatformTypeEnum getType() {
        return PlatformTypeEnum.HUAWEI;
    }

    @Override
    public void run(ApplicationArguments args) {
        if (app == null) {
            HuaweiCredential credential = HuaweiCredential.builder()
                    .setAppId(appPushProperties.getHuaweiAppId())
                    .setAppSecret(appPushProperties.getHuaweiClientSecret())
                    .build();

            HuaweiOption option = HuaweiOption.builder()
                    .setCredential(credential)
                    .build();

            app = HuaweiApp.initializeApp(option);
            logger.debug("初始化华为App");
        }
    }
}
