package com.bxm.localnews.msg.platform.impl;

import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.constant.PushFoldTypeEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.push.PushResponseCallback;
import com.bxm.localnews.msg.push.Response;
import com.bxm.localnews.msg.vo.MsgUserTokenBean;
import com.google.common.collect.Lists;
import com.xiaomi.xmpush.server.*;
import org.apache.commons.lang3.StringUtils;
import org.json.simple.parser.ParseException;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.List;
import java.util.Objects;

import static com.alibaba.fastjson.JSON.toJSONString;

@Component
@RefreshScope
public class MiuiPushPlatform extends AbstractPushPlatform {


    @Override
    public void push(PushMessage message, MsgUserTokenBean userToken, PushResponseCallback callback) {
        Constants.useOfficial();
        Sender sender = new Sender(appPushProperties.getMiuiAppSecret());

        String title = message.getTitle();
        if (title.length() > 50) {
            title = title.substring(0, 47) + "...";
        } else if (StringUtils.isBlank(title)) {
            title = "推送通知";
        }

        int maxContentLength = 128;

        String description = message.getContent();
        if (Objects.nonNull(description) && description.length() > maxContentLength) {
            description = description.substring(0, maxContentLength);
        } else if (StringUtils.isBlank(description)) {
            description = title;
        }
        message.getPayloadInfo().getExtend().put("notifyImageUrl", message.getNotifyImageUrl());
        message.getPayloadInfo().getExtend().put("notifyText", message.getNotifyText());

        Message.Builder messageBuilder = new Message.Builder()
                .title(title)
                .description(description)
                .timeToLive(message.getPeriodTime() * 3600000L)
                .payload(toJSONString(message.getPayloadInfo()))
                .restrictedPackageName(appPushProperties.getAppPkgName());

        fillMessageBuilder(messageBuilder, message);

        if (PushFoldTypeEnum.IMAGE.equals(message.getPushFoldType())) {
            File file = new File(message.getNotifyImageUrl());
            Media media = new Media(this.appPushProperties.getMiuiAppSecret());
            String picUrl = null;
            try {
                Result result = media.upload(file, false, true);

                if (logger.isDebugEnabled()) {
                    logger.debug("小米上传推送文件返回结果：[{}], 返回的数据：[{}]",
                            result.getErrorCode(),
                            result.getData().toJSONString());
                }

                picUrl = URLDecoder.decode(result.getData(Constants.JSON_MEDIA_PICTURE_URL), "UTF8");
            } catch (IOException e) {
                logger.error("小米推送图文消息上传大图错误: 图片地址：[{}]", message.getNotifyImageUrl());
            }

            messageBuilder.extra(new Message.NotificationStyleBuilder().bigPictureStyle()
                    .bigPicUri(picUrl).build());
        }

        if (PushFoldTypeEnum.TEXT.equals(message.getPushFoldType())) {
            messageBuilder.extra(new Message.NotificationStyleBuilder().bigTextStyle().build());
        }

        try {
            List<String> regIdList = Lists.newArrayList(userToken.getToken());
            Result result = sender.send(messageBuilder.build(), regIdList, 3);

            String parseResult = parseResult(result);
            if (SUCCESS.equals(parseResult)) {
                callback.apply(message, userToken, Response.success());
            } else {
                logger.info("推送失败，重新推送。消息:[{}],失败原因：[{}]", message, parseResult);
                callback.apply(message, userToken, Response.fail(parseResult));
            }
        } catch (IOException | ParseException e) {
            callback.apply(message, userToken, Response.fail(e.getMessage()));
            logger.error(e.getMessage(), e);
        }
    }

    private void fillMessageBuilder(Message.Builder messageBuilder, PushMessage message) {
        if (message.getActionType() != null) {
            String encodeMsg = "";
            try {
                encodeMsg = URLEncoder.encode(message.getPayloadInfo().toJsonString(), "UTF-8");
            } catch (UnsupportedEncodingException e) {
                logger.error(e.getMessage(), e);
            }
            String intent = "intent:#Intent;component=com.bxm.localmastercard/.ui.push.PushRouteActivity;S.message=" + encodeMsg +
                    ";end";

            if (logger.isDebugEnabled()) {
                logger.debug("推送报文:{}", intent);
            }
            messageBuilder.extra(Constants.EXTRA_PARAM_INTENT_URI, intent);
        }

        //据安卓开发[鲍振东]所言，小米目前支持不了通知，故改为透传（又改回来了）
        if (TemplateTypeEnum.NOTIFCTION.equals(message.getType())) {
            messageBuilder.passThrough(0);
        } else {
            messageBuilder.passThrough(1);
        }

        if (message.isMute()) {
            messageBuilder.notifyType(0);
        } else if (message.isShock()) {
            messageBuilder.notifyType(1 | 2);
        } else {
            messageBuilder.notifyType(1);
        }
    }

    private String parseResult(Result result) {
        if (result.getErrorCode().getValue() == 0) {
            return SUCCESS;
        } else {
            return result.getReason();
        }
    }

    @Override
    public PlatformTypeEnum getType() {
        return PlatformTypeEnum.MIUI;
    }
}
