package com.bxm.localnews.msg.push;

import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.config.AppPushProperties;
import com.bxm.localnews.msg.platform.IPushPlatform;
import com.bxm.localnews.msg.push.impl.DefaultPushResponseCallback;
import com.bxm.localnews.msg.rule.IPushRule;
import com.bxm.localnews.msg.service.UserMessageService;
import com.bxm.localnews.msg.service.UserTokenService;
import com.bxm.localnews.msg.vo.MsgUserTokenBean;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Map;

/**
 * 推送的最终执行者，负责对消息进行规则校验
 * 推送消息只支持单条消息
 *
 * @author liujia
 * @date 2020-07-05 10:42
 **/
@Component
@Slf4j
public class PushExecutor {

    private final List<IPushRule> rules;

    private final UserTokenService userTokenService;

    private final UserMessageService userMessageService;

    private Map<PlatformTypeEnum, IPushPlatform> platformMap;

    private final DefaultPushResponseCallback pushResponseCallback;

    private final AppPushProperties appPushProperties;

    /**
     * 默认的推送平台，当无法匹配推送的时候，进行兜底处理，可有可无
     * 主要是辅助进行测试工作
     * 也可以将无法推送的消息放入到特殊记录等情况处理
     */
    private IPushPlatform defaultPlatform;

    @Autowired
    public PushExecutor(List<IPushRule> rules,
                        UserTokenService userTokenService,
                        UserMessageService userMessageService,
                        List<IPushPlatform> platforms,
                        DefaultPushResponseCallback pushResponseCallback,
                        AppPushProperties appPushProperties) {
        this.rules = rules;
        this.userTokenService = userTokenService;
        this.userMessageService = userMessageService;
        this.pushResponseCallback = pushResponseCallback;
        this.appPushProperties = appPushProperties;

        platformMap = Maps.newHashMap();
        for (IPushPlatform platform : platforms) {
            platformMap.put(platform.getType(), platform);
        }
    }

    public void setDefaultPlatform(IPushPlatform defaultPlatform) {
        this.defaultPlatform = defaultPlatform;
    }

    /**
     * 执行消息推送
     * 通过规则链条，判断消息是否合法
     * 获取用户信息，匹配推送平台，执行最终的消息送达
     * 如果是待统计消息，根据消息推送的结果更新统计数据
     * 将消息写入到用户消息中
     *
     * @param message 推送消息
     */
    public void push(PushMessage message) {
        boolean passRule = true;

        for (IPushRule rule : rules) {
            passRule = rule.accpect(message);
            if (!passRule) {
                // 验证失败，但是继续写入到消息中心
                if (rule.write()) {
                    break;
                }

                if (log.isDebugEnabled()) {
                    log.debug("消息发送验证失败，验证规则：{},推送内容：{}", rule.getClass(), message);
                }
                return;
            }
        }

        // 获取用户推送相关的信息
        Long userId = message.getPushReceiveScope().getUserId();
        saveMessage(message, userId);

        if (passRule) {
            try {
                List<MsgUserTokenBean> tokenList = userTokenService.loadToken(userId);

                // 匹配对应的推送平台，执行推送信息
                boolean match = false;
                for (MsgUserTokenBean userTokenBean : tokenList) {
                    if (null == userTokenBean.getToken()) {
                        continue;
                    }
                    PlatformTypeEnum platform = PlatformTypeEnum.getByCode(userTokenBean.getType());

                    if (platform.isApp()) {
                        match = true;

                        fillMessage(message);

                        platformMap.get(platform).push(message, userTokenBean, pushResponseCallback);
                        break;
                    }
                }


                if (!match && defaultPlatform != null) {
                    MsgUserTokenBean token = new MsgUserTokenBean();
                    token.setUserId(userId);
                    defaultPlatform.push(message, token, pushResponseCallback);
                }
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }
    }

    /**
     * 填充推送的默认字段，部分平台必须要有title和content
     *
     * @param message 推送消息
     */
    private void fillMessage(PushMessage message) {
        //设置默认内容
        if (StringUtils.isEmpty(message.getContent())) {
            message.setContent(appPushProperties.getDefaultPushContent());
        }

    }

    private void saveMessage(PushMessage message, Long userId) {
        if (message.isPersistence()) {
            // 将需要序列化的消息写入到用户消息中心，增加用户未读消息数量
            userMessageService.addMessage(message, userId);
        }
    }
}


















