package com.bxm.localnews.msg.platform.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.httpclient.service.HttpClientService;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.config.AppPushProperties;
import com.bxm.localnews.msg.platform.impl.umeng.AndroidNotification;
import com.bxm.localnews.msg.platform.impl.umeng.UmengNotification;
import com.bxm.localnews.msg.platform.impl.umeng.android.AndroidUnicast;
import com.bxm.localnews.msg.platform.impl.umeng.ios.AlertMsg;
import com.bxm.localnews.msg.platform.impl.umeng.ios.IOSUnicast;
import com.bxm.localnews.msg.push.PushResponseCallback;
import com.bxm.localnews.msg.push.Response;
import com.bxm.localnews.msg.vo.MsgUserTokenBean;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import org.apache.commons.codec.digest.DigestUtils;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Calendar;
import java.util.Date;
import java.util.Objects;

import static com.bxm.localnews.mq.common.constant.PushSoundEnum.IOS_MUTE;

/**
 * 友盟推送平台：https://developer.umeng.com/docs/67966/detail/68343
 */
@Component
@RefreshScope
public class UmengPushPlatform extends AbstractPushPlatform {

    private static final String HOST = "http://msg.umeng.com/api/send";

    @Resource
    private HttpClientService httpClientService;

    @Resource
    private AppPushProperties appPushProperties;

    @Override
    public PlatformTypeEnum getType() {
        return PlatformTypeEnum.UMENG;
    }

    @Override
    public void push(PushMessage message, MsgUserTokenBean userToken, PushResponseCallback callback) {
        logger.debug("消息推送类型为：[{}]", message.getType());

        UmengNotification unicast;
        if (Objects.equals(PlatformEnum.IOS.getCode(), Integer.valueOf(userToken.getPlatform()))) {
            unicast = createIOSUnicast(message, userToken);
        } else {
            unicast = createAndroidUnicast(message, userToken);
        }

        if (null == unicast) {
            callback.apply(message, userToken, Response.fail("nullpoint"));
            return;
        }

        String postUrl = HOST;
        String postBody = unicast.getPostBody();
        String sign = StringUtils.EMPTY;

        logger.debug("推送消息：[{}]", postBody);

        try {
            sign = DigestUtils.md5Hex(("POST" + postUrl + postBody + unicast.getAppMasterSecret()).getBytes("utf8"));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        postUrl = postUrl + "?sign=" + sign;

        try {
            String response = httpClientService.doPost(postUrl, postBody);
            logger.debug("umeng push result:[{}]", response);

            if (null == response) {
                callback.apply(message, userToken, Response.fail("without response"));
                return;
            }

            String result = parseResult(response);
            if (SUCCESS.equals(result)) {
                //推送完成回调
                callback.apply(message, userToken, Response.success());
            } else {
                callback.apply(message, userToken, Response.fail(result));
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            callback.apply(message, userToken, Response.fail(e.getMessage()));
        }
    }

    private String parseResult(String response) {
        JSONObject json = JSONObject.parseObject(response);
        String ret = json.getString("ret");
        if (StringUtils.equalsIgnoreCase("SUCCESS", ret)) {
            return SUCCESS;
        } else {
            return response;
        }
    }

    /**
     * 创建ios单播推送消息
     */
    private IOSUnicast createIOSUnicast(PushMessage message, MsgUserTokenBean userToken) {
        try {
            IOSUnicast unicast = new IOSUnicast();

            unicast.setAppMasterSecret(appPushProperties.getUmengIosAppSecrect());
            unicast.setPredefinedKeyValue("appkey", appPushProperties.getUmengIosAppkey());

            String timestamp = Integer.toString((int) (System.currentTimeMillis() / 1000));
            unicast.setPredefinedKeyValue("timestamp", timestamp);

            unicast.setDeviceToken(userToken.getToken());
            AlertMsg alertMsg = new AlertMsg(message.getTitle(), message.getContent());
            unicast.setAlert(JSONObject.toJSON(alertMsg));

            unicast.setContentAvailable(0);
            unicast.setBadge(1);

            if (appPushProperties.isUmengProductionMode()) {
                unicast.setProductionMode();
            } else {
                unicast.setTestMode();
            }

            if (message.isMute()) {
                unicast.setSound(IOS_MUTE.getIosSound());
            } else {
                unicast.setSound(message.getSound().getIosSound());
            }

            //组装payload
            unicast.setCustomizedField("extend", JSONObject.toJSON(message.getPayloadInfo().getExtend()));
            unicast.setCustomizedField("type", message.getPayloadInfo().getType());
            unicast.setCustomizedField("title", message.getTitle());
            unicast.setCustomizedField("content", message.getContent());
            unicast.setCustomizedField("msgId", message.getPayloadInfo().getMsgId());
            if (message.getPeriodTime() != null) {
                Date now = new Date();
                Date expireDate = DateUtils.addField(now, Calendar.HOUR, message.getPeriodTime());
                unicast.setExpireTime(DateUtils.formatAtWill(expireDate, DateUtils.DATE_TIME_FORMAT));
            }

            return unicast;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return null;
        }
    }

    /**
     * 创建安卓单播消息
     */
    private AndroidUnicast createAndroidUnicast(PushMessage message, MsgUserTokenBean userToken) {
        try {
            AndroidUnicast unicast = new AndroidUnicast();

            unicast.setAppMasterSecret(appPushProperties.getUmengAndroidAppSecrect());
            unicast.setPredefinedKeyValue("appkey", appPushProperties.getUmengAndroidAppkey());

            String timestamp = Integer.toString((int) (System.currentTimeMillis() / 1000));
            unicast.setPredefinedKeyValue("timestamp", timestamp);

            unicast.setDeviceToken(userToken.getToken());
            //设置通知内容
            unicast.setBody(message.getTitle(), message.getTitle(), message.getContent());

            //设置提醒方式
            unicast.setPlay(message.isShock(), message.isMute());
            if (message.isMute()) {
                unicast.setSound(IOS_MUTE.getAndriodSound());
            } else {
                unicast.setSound(message.getSound().getAndriodSound());
            }

            if (TemplateTypeEnum.TRANSMISSION.equals(message.getType())) {
                unicast.setDisplayType(AndroidNotification.DisplayType.MESSAGE);
            } else {
                unicast.setDisplayType(AndroidNotification.DisplayType.NOTIFICATION);
            }
            unicast.setCustomField(JSONObject.parseObject(JSONObject.toJSONString(message.getPayloadInfo())));
            //设置默认打开app
            unicast.goAppAfterOpen();

            if (appPushProperties.isUmengProductionMode()) {
                unicast.setProductionMode();
            } else {
                unicast.setTestMode();
            }
            unicast.setExtra(JSONObject.parseObject(JSON.toJSONString(message.getPayloadInfo())));
            if (message.getPeriodTime() != null) {
                Date now = new Date();
                Date expireDate = DateUtils.addField(now, Calendar.HOUR, message.getPeriodTime());
                unicast.setExpireTime(DateUtils.formatAtWill(expireDate, DateUtils.DATE_TIME_FORMAT));
            }
            return unicast;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return null;
        }
    }
}
