package com.bxm.localnews.msg.platform.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.msg.push.PushResponseCallback;
import com.bxm.localnews.msg.push.Response;
import com.bxm.localnews.msg.vo.MsgUserTokenBean;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.UUIDUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.vivo.push.sdk.notofication.Message;
import com.vivo.push.sdk.notofication.Result;
import com.vivo.push.sdk.server.Sender;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import static com.bxm.localnews.msg.constant.RedisConfig.VIVO_ACCESS_TOKEN;
import static com.bxm.localnews.msg.constant.RedisConfig.VIVO_ACCESS_TOKEN_EXPIRED;

/**
 * vivo推送平台
 * 相关文档：https://dev.vivo.com.cn/documentCenter/doc/155
 *
 * @author liujia
 * @date 2019/12/4 16:35
 */
@Component
@Slf4j
public class VivoPushPlatform extends AbstractPushPlatform {

    private Sender sender;

    private AtomicBoolean init = new AtomicBoolean(false);

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private DistributedLock distributedLock;

    @Resource
    private SequenceCreater sequenceCreater;

    /**
     * 鉴权token的默认过期时间(2小时)
     */
    private static final Long DEFAULT_EXPIRED_SECONDS = 2 * 60 * 60L;

    /**
     * 最长的传输内容
     */
    private static final int MAX_SKIP_CONTENT_LEN = 1024;

    /**
     * 分布式锁的资源key
     */
    private static final String LOCK_RESOURCE_KEY = "vivoToken";

    /**
     * vivo推送成功的状态码
     */
    private static final int SUCCESS_CODE = 0;

    /**
     * 用户token或别名已过期
     */
    private static final Integer ALIAS_EXPIRED = 10307;

    private Cache<String, String> tokenCache;

    private String tokenCacheKey = "token";

    @Override
    public void push(PushMessage message, MsgUserTokenBean userToken, PushResponseCallback callback) {
        if (null == tokenCache || tokenCache.getIfPresent(tokenCacheKey) == null) {
            initSender();
            refreshToken(3);
        }

        Message.Builder messageBuilder = getBuilder(message);
        messageBuilder.regId(userToken.getToken());

        try {
            Result result = sender.sendSingle(messageBuilder.build());

            if (SUCCESS_CODE == result.getResult()) {
                callback.apply(message, userToken, Response.success());
            } else if (ALIAS_EXPIRED == result.getResult()) {
                // 用户token已过期
                callback.apply(message, userToken, Response.expire(result.getDesc()));
            } else {
                // 其他推送失败的情况
                callback.apply(message, userToken, Response.fail(result.getDesc()));
            }
        } catch (IllegalArgumentException e) {
            if (StringUtils.startsWith(e.getMessage(), "ParameterError: 10302")) {
                callback.apply(message, userToken, Response.expire(e.getMessage()));
            } else {
                callback.apply(message, userToken, Response.fail(e.getMessage()));
            }
            log.error(e.getMessage(), e);
        } catch (Exception e) {
            callback.apply(message, userToken, Response.fail(e.getMessage()));
            log.error(e.getMessage(), e);
        }
    }

    private Message.Builder getBuilder(PushMessage pushMessage) {
        return new Message.Builder()
                .title(subString(pushMessage.getTitle(), 40))
                .content(subString(pushMessage.getTitle(), 50))
                .timeToLive(convertLiveSeconds(pushMessage))
                .notifyType(convertNotifyType(pushMessage))
                .networkType(-1)
                .requestId(sequenceCreater.nextStringId())
                .skipType(4)
                .skipContent(buildSkipContent(pushMessage));
    }

    /**
     * 裁剪内容，使其符合推送平台的字符限制
     *
     * @param content      内容
     * @param expectLength 预期长度
     * @return 裁剪结果
     */
    private String subString(String content, int expectLength) {
        int length = caculateCharSize(content);
        if (length > expectLength) {
            return StringUtils.substring(content, 0, expectLength / 2);
        }

        return content;
    }

    private int caculateCharSize(String str) {
        if (str == null) {
            return 0;
        } else {
            int chineseCharSize = 0;

            for (int i = 0; i < str.length(); ++i) {
                if (isChinese(str.charAt(i))) {
                    ++chineseCharSize;
                }
            }

            return str.length() + chineseCharSize;
        }
    }

    private boolean isChinese(char c) {
        Character.UnicodeBlock ub = Character.UnicodeBlock.of(c);
        if (ub == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS) {
            return true;
        } else if (ub == Character.UnicodeBlock.CJK_COMPATIBILITY_IDEOGRAPHS) {
            return true;
        } else if (ub == Character.UnicodeBlock.CJK_SYMBOLS_AND_PUNCTUATION) {
            return true;
        } else if (ub == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A) {
            return true;
        } else if (ub == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B) {
            return true;
        } else if (ub == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C) {
            return true;
        } else if (ub == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_D) {
            return true;
        } else if (ub == Character.UnicodeBlock.GENERAL_PUNCTUATION) {
            return true;
        } else {
            return ub == Character.UnicodeBlock.HALFWIDTH_AND_FULLWIDTH_FORMS;
        }
    }

    /**
     * 自定义传输内容不可超过1024个字符
     *
     * @param pushMessage 推送内容
     * @return 传输内容
     */
    private String buildSkipContent(PushMessage pushMessage) {
        PushPayloadInfo payloadInfo = pushMessage.getPayloadInfo();
        payloadInfo.setTitle(null);
        payloadInfo.setContent(null);

        String skipContent = payloadInfo.toJsonString();

        if (StringUtils.length(skipContent) > MAX_SKIP_CONTENT_LEN) {
            log.warn("传输内容超过1024个字符，进行了截断，保证能正常传输.原始内容：{}", skipContent);

            //删除非必要字段
            JSONObject jsonObj = JSON.parseObject(skipContent);
            jsonObj.remove("createTime");
            jsonObj.remove("hasDetail");
            jsonObj.remove("sound");
            jsonObj.remove("title");
            jsonObj.remove("content");

            skipContent = jsonObj.toJSONString();
            //如果长度还超过1024，则删除跳转协议和额外参数
            if (StringUtils.length(skipContent) > MAX_SKIP_CONTENT_LEN) {
                log.error("推送内容仍然超长，删除协议字段和额外参数，裁剪后的结果：{}", skipContent);

                jsonObj.remove("protocol");
                jsonObj.remove("extend");

                skipContent = jsonObj.toJSONString();
            }
        }

        return skipContent;
    }

    /**
     * 推送保留时长
     *
     * @param pushMessage 推送消息
     * @return 消息在推送平台存活时间
     */
    private int convertLiveSeconds(PushMessage pushMessage) {
        if (null != pushMessage.getPeriodTime() && pushMessage.getPeriodTime() > 0) {
            return pushMessage.getPeriodTime() * 3600;
        }
        return 7 * 24 * 3600;
    }

    /**
     * 根据推送参数转换API要求的通知类型
     *
     * @param message 推送消息
     * @return 转换结果
     */
    private int convertNotifyType(PushMessage message) {
        if (!message.isMute() && message.isShock()) {
            return 4;
        }

        if (message.isShock()) {
            return 3;
        }

        if (!message.isMute()) {
            return 2;
        }
        return 1;
    }

    @Override
    public PlatformTypeEnum getType() {
        return PlatformTypeEnum.VIVO;
    }

    /**
     * 创建本地缓存缓存,设置指定时间后过去，获取不到token后，从redis加载
     *
     * @param token             新的授权token值
     * @param expredMillseconds 过期的毫秒数
     */
    private void initLocalCache(String token, long expredMillseconds) {
        // 创建本地缓存
        Cache<String, String> swapCache = CacheBuilder.newBuilder()
                .initialCapacity(1)
                .expireAfterWrite(expredMillseconds, TimeUnit.MILLISECONDS)
                .build();

        swapCache.put(tokenCacheKey, token);
        tokenCache = swapCache;
    }

    private void initSender() {
        if (init.compareAndSet(false, true)) {
            try {
                sender = new Sender(appPushProperties.getVivoAppSecret());
                sender.initPool(20, 10);
                log.info("初始化vivo token");

            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }
    }

    private void refreshToken(int maxAttempts) {
        if (maxAttempts <= 0) {
            logger.error("vivo重试获取token失败");
            return;
        }

        // 尝试从redis加载
        String authToken = redisStringAdapter.get(VIVO_ACCESS_TOKEN, String.class);
        Long expiredMills = redisStringAdapter.get(VIVO_ACCESS_TOKEN_EXPIRED, Long.class);

        if (null == authToken || expiredMills < System.currentTimeMillis()) {
            String unlockKey = UUIDUtils.nextID();
            if (distributedLock.lock(LOCK_RESOURCE_KEY, unlockKey)) {
                try {
                    Result token = sender.getToken(appPushProperties.getVivoAppId(), appPushProperties.getVivoAppKey());
                    authToken = token.getAuthToken();
                    if (StringUtils.isBlank(authToken)) {
                        logger.error("get vivo token failed");
                        refreshToken(--maxAttempts);
                        return;
                    }

                    //设置内存存储过期时间，比官方文档提前一小时过期，防止过期与文档描述不符
                    expiredMills = System.currentTimeMillis() + 5400 * 1000L;

                    redisStringAdapter.set(VIVO_ACCESS_TOKEN, authToken, DEFAULT_EXPIRED_SECONDS);
                    redisStringAdapter.set(VIVO_ACCESS_TOKEN_EXPIRED, expiredMills);
                    log.info("refresh vivo token,token:{},expires time:{}", authToken, expiredMills);

                    distributedLock.unlock(LOCK_RESOURCE_KEY, unlockKey);
                } catch (Exception e) {
                    log.error(e.getMessage(), e);
                    refreshToken(--maxAttempts);
                    return;
                }
            } else {
                try {
                    Thread.sleep(500L);
                    log.info("retry refresh vivo token");
                    refreshToken(--maxAttempts);
                    return;
                } catch (InterruptedException e) {
                    log.error(e.getMessage(), e);
                    Thread.currentThread().interrupt();
                }
            }
        }

        initLocalCache(authToken, expiredMills - System.currentTimeMillis());
        sender.setAuthToken(authToken);
    }

}
