package com.bxm.localnews.msg.push.impl;

import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.push.PushResponseCallback;
import com.bxm.localnews.msg.push.Response;
import com.bxm.localnews.msg.service.MessageGroupCounterService;
import com.bxm.localnews.msg.service.MessageGroupService;
import com.bxm.localnews.msg.service.UserTokenService;
import com.bxm.localnews.msg.vo.MsgGroupPushCounterBean;
import com.bxm.localnews.msg.vo.MsgUserTokenBean;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;

/**
 * 推送处理完成后的回调处理
 * 对推送失败、推送成功、推送用户失效进行不同的处理
 *
 * @author liujia
 * @date 2020-07-06 19:14
 **/
@Component
@Slf4j
public class DefaultPushResponseCallback implements PushResponseCallback {

    @Resource
    private MessageGroupService messageGroupService;

    @Resource
    private UserTokenService userTokenService;

    @Resource
    private MessageGroupCounterService messageGroupCounterService;

    @Override
    public void apply(PushMessage message, MsgUserTokenBean userToken, Response response) {

        if (log.isDebugEnabled()) {
            log.debug("收到推送回调信息，消息ID：{},token:{},response:{}",
                    message.getMsgId(),
                    userToken,
                    response);
        }

        if (response.isTokenExpired()) {
            // 将用户的token设置为过期状态
            userTokenService.removeToken(userToken);
        }

        if (response.isSuccess()) {
            success(message, userToken);
        } else {
            failed(message, userToken, response);
        }
    }

    private void success(PushMessage message, MsgUserTokenBean userToken) {
        if (messageGroupService.isGroupMsg(message.getMsgId())) {
            // 增加推送成功的统计数量
            messageGroupCounterService.addCounter(MsgGroupPushCounterBean.builder()
                    .groupMsgId(message.getMsgId())
                    .success(1)
                    .pushType(userToken.getType())
                    .build());

        }
    }

    private void failed(PushMessage message, MsgUserTokenBean userToken, Response response) {
        if (messageGroupService.isGroupMsg(message.getMsgId())) {
            //增加推送失败的统计数量
            messageGroupCounterService.addCounter(MsgGroupPushCounterBean.builder()
                    .groupMsgId(message.getMsgId())
                    .fail(1)
                    .pushType(userToken.getType())
                    .build());
        }

        if (log.isDebugEnabled()) {
            log.debug("消息推送失败，失败原因：{},token : {}", response.getMessage(), userToken.getToken());
        }
    }
}
