package com.bxm.localnews.msg.strategy.impl;

import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.config.AppPushProperties;
import com.bxm.localnews.msg.config.PushMessageStatusEnum;
import com.bxm.localnews.msg.param.PushMessageBucket;
import com.bxm.localnews.msg.service.MessageGroupCounterService;
import com.bxm.localnews.msg.service.MessageGroupService;
import com.bxm.localnews.msg.strategy.IPushDistributeStrategy;
import com.bxm.localnews.msg.stream.MessageProvider;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.extern.slf4j.Slf4j;

import javax.annotation.Resource;
import java.util.List;

/**
 * 推送分发策略的抽象类
 *
 * @author liujia
 * @date 2020-07-04 12:17
 **/
@Slf4j
public abstract class AbstractPushDistributeStrategy implements IPushDistributeStrategy {

    private MessageGroupService messageGroupService;

    @Resource
    private MessageGroupCounterService messageGroupCounterService;

    @Resource
    protected AppPushProperties appPushProperties;

    @Resource
    protected MessageProvider messageProvider;

    protected MessageGroupService getMessageGroupService() {
        if (messageGroupService == null) {
            messageGroupService = SpringContextHolder.getBean(MessageGroupService.class);
        }

        return messageGroupService;
    }

    /**
     * 如果推送消息需要进行发送情况、已读情况统计，进行数据预处理
     *
     * @param message 推送的消息内容
     */
    void execPreprocess(PushMessage message) {
        // 写入统计表，方便后续调用更新
        messageGroupCounterService.saveCounter(message);
        // 消息状态更新为发送中
        getMessageGroupService().changeStatus(message.getMsgId(), PushMessageStatusEnum.BEING_SEND);
    }

    /**
     * 拆分推送任务，将其拆解为更小的单元
     *
     * @param userIds 待推送的用户ID
     * @param msgId   推送的消息ID
     * @param index   群推分片的当前索引
     * @param total   总分片数量
     */
    void splitTask(List<Long> userIds, Long msgId, int index, int total) {
        // 根据分桶数量进行拆分
        int bucketNum = appPushProperties.getBucketNum();
        int size = userIds.size();
        int splitNum = (size / bucketNum) + (size % bucketNum == 0 ? 0 : 1);


        for (int i = 0; i < splitNum; i++) {
            int startIndex = i * bucketNum;
            int endIndex = startIndex + bucketNum;

            if (endIndex > size) {
                endIndex = size;
            }
            List<Long> subUserIds = userIds.subList(startIndex, endIndex);

            // 构建统计通道需要的数据结构
            PushMessageBucket param = PushMessageBucket.builder()
                    .pushMsgId(msgId)
                    .targetUserIds(subUserIds)
                    .index(index)
                    .total(total)
                    .build();

            if (log.isDebugEnabled()) {
                log.debug("msgId:{},index:{},total:{}", msgId, index, total);
            }
            messageProvider.batchPush(param);
        }

        // 增加消息的预计触达人数
        getMessageGroupService().addPushTotal(msgId, userIds.size());
    }
}
