package com.bxm.localnews.mq.consume.push;

import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.mq.consume.push.platform.IPushPlatform;
import com.bxm.localnews.mq.consume.push.receive.IReceiveUserIterator;
import com.bxm.localnews.mq.consume.push.receive.ReceiveUserFactory;
import com.bxm.localnews.msg.domain.PushMessageMapper;
import com.bxm.localnews.msg.vo.UserDeviceBean;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;

/**
 * 消息推送调度中心
 * 根据推送消息的接收者获取对应的接收用户列表
 * 将用户根据设备类型进行区分，根据具体的平台执行推送
 * @author liujia
 */
@Component
public class PushMessageDispatcher implements ApplicationListener<ContextRefreshedEvent>, ApplicationContextAware {

    private static final Logger LOGGER = LoggerFactory.getLogger(PushMessageDispatcher.class);

    private final ReceiveUserFactory receiveUserManage;

    private final AsyncTaskExecutor asyncTaskExecutor;

    private final PushMessageMapper pushMessageMapper;

    private Map<PlatformTypeEnum, IPushPlatform> pushPlatformMap;

    private ApplicationContext applicationContext;

    @Autowired
    public PushMessageDispatcher(ReceiveUserFactory receiveUserManage, AsyncTaskExecutor asyncTaskExecutor,
                                 PushMessageMapper pushMessageMapper) {
        this.receiveUserManage = receiveUserManage;
        this.asyncTaskExecutor = asyncTaskExecutor;
        this.pushMessageMapper = pushMessageMapper;
    }

    /**
     * 执行具体的推送
     * 1.根据用户推送范围获取到用户信息拉取的迭代器
     * 2.根据迭代器获取推送批次的用户
     * 3.异步进行推送动作
     * @param message 推送消息参数
     */
    public void push(PushMessage message) {
        LOGGER.debug("消息推送开始,message:{}", message);

        PushReceiveScope receiveScope = message.getPushReceiveScope();
        //根据消息接收范围获取对应的用户信息
        IReceiveUserIterator iterator = receiveUserManage.get(receiveScope.getPushReceiverRule(), receiveScope.getRuleParam());

        List<PlatformTypeEnum> platformTypes = message.getPushReceiveScope().getPlatformTypes();

        while (iterator.hasNext()) {
            Map<PlatformTypeEnum, List<UserDeviceBean>> userMap = iterator.next();

            asyncTaskExecutor.execute(() -> userMap.forEach((type, users) -> {
                if (null != platformTypes) {
                    for (PlatformTypeEnum platform : platformTypes) {
                        if (platform.equals(type)) {
                            pushPlatformMap.get(type).push(message, users);
                        }
                    }
                } else {
                    pushPlatformMap.get(type).push(message, users);
                }
            }));
        }

        //完成推送后将消息更新为推送完成
        pushMessageMapper.updateStats(message.getPayloadInfo().getMsgId(), 2);

        LOGGER.debug("消息推送结束,message:{}", message);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void onApplicationEvent(ContextRefreshedEvent event) {
        Map<String, IPushPlatform> pushPlatormBeanMap = applicationContext.getBeansOfType(IPushPlatform.class);
        if (pushPlatormBeanMap.size() == 0) {
            throw new IllegalArgumentException("push platform is emtpy");
        }
        pushPlatformMap = Maps.newHashMap();
        pushPlatormBeanMap.forEach((key, value) -> {
            pushPlatformMap.put(value.getType(), value);
        });

    }
}
