package com.bxm.localnews.mq.consume.push.platform.impl;

import com.bxm.component.mybatis.utils.BatchHelper;
import com.bxm.localnews.mq.common.constant.PushReceiverRuleEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.consume.push.platform.IPushPlatform;
import com.bxm.localnews.msg.domain.MessageMapper;
import com.bxm.localnews.msg.domain.PushMessageMapper;
import com.bxm.localnews.msg.domain.PushUserMapper;
import com.bxm.localnews.msg.vo.PushUserBean;
import com.bxm.localnews.msg.vo.UserDeviceBean;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;
import java.util.List;

/**
 * 抽象的推送平台类
 * 处理推送相关的共同内容
 * @author liujia
 */
public abstract class AbstractPushPlatform implements IPushPlatform {

    Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private PushUserMapper pushUserMapper;

    @Autowired
    private PushMessageMapper pushMessageMapper;

    @Autowired
    private MessageMapper messageMapper;

    @Override
    public void push(PushMessage message, List<UserDeviceBean> userDevice) {
        if (userDevice.size() > 0) {
            if (userDevice.size() == 1) {
                singlePush(message, userDevice.get(0));
            } else {
                groupPush(message, userDevice);
            }
            changePushNum(message, userDevice);
        }
    }

    /**
     * 推送完成后更新用户触达人数
     * 非默认（即指定单人推送或固定人数推送的情况）才需要进行人数的更新
     * @param message    消息信息
     * @param userDevice 用户列表
     */
    private void changePushNum(PushMessage message, List<UserDeviceBean> userDevice) {
        if (!PushReceiverRuleEnum.DEFAULT.equals(message.getPushReceiveScope().getPushReceiverRule())) {
            pushMessageMapper.updateNum(message.getPayloadInfo().getMsgId(), userDevice.size());
        }
    }

    /**
     * 推送结果回调处理
     */
    void callback(PushMessage message, UserDeviceBean user, String response) {
        callback(message, ImmutableList.of(user), response);
    }

    void callback(PushMessage message, List<UserDeviceBean> users, String response) {
        new BatchHelper<PushUserMapper, UserDeviceBean>(PushUserMapper.class, users) {

            @Override
            protected int invoke(UserDeviceBean element) {
                PushUserBean entity = new PushUserBean();
                entity.setId(sequenceCreater.nextLongId());
                entity.setChecked(0);
                entity.setMessageId(message.getPayloadInfo().getMsgId());
                entity.setPlatform(element.getPlatform());
                entity.setUserId(element.getUserId());
                entity.setPushTime(new Date());
                entity.setPushResult(response);
                return mapper.insert(entity);
            }
        };

        if (message.isPersistence()) {

        }
    }

    /**
     * 推送消息到单独的用户
     * @param message    消息体
     * @param userDevice 用户设备信息
     */
    abstract void singlePush(PushMessage message, UserDeviceBean userDevice);

    /**
     * 推送消息到群组
     * @param message    消息体
     * @param userDevice 用户设备信息列表
     */
    abstract void groupPush(PushMessage message, List<UserDeviceBean> userDevice);

    /**
     * 判断用户设备是否iOS
     * @param userDevice 用户设备信息
     * @return
     */
    boolean isIOS(UserDeviceBean userDevice) {
        return null != userDevice && null != userDevice.getPhoneModel() && userDevice.getPhoneModel().indexOf("iPhone") > 0;
    }
}
