package com.bxm.localnews.mq.consume.push.platform.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.config.AppPushProperties;
import com.bxm.localnews.msg.vo.UserDeviceBean;
import com.bxm.newidea.component.tools.StringUtils;
import com.gexin.rp.sdk.base.IPushResult;
import com.gexin.rp.sdk.base.ITemplate;
import com.gexin.rp.sdk.base.impl.SingleMessage;
import com.gexin.rp.sdk.base.impl.Target;
import com.gexin.rp.sdk.base.payload.APNPayload;
import com.gexin.rp.sdk.http.IGtPush;
import com.gexin.rp.sdk.template.NotificationTemplate;
import com.gexin.rp.sdk.template.TransmissionTemplate;
import com.gexin.rp.sdk.template.style.Style0;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;

import static com.bxm.localnews.mq.common.constant.PushSoundEnum.IOS_MUTE;

/**
 * 个推推送平台接入
 */
@Component
public class GetuiPushPlatform extends AbstractPushPlatform {

    private static final Logger LOGGER = LoggerFactory.getLogger(GetuiPushPlatform.class);

    @Resource
    private AppPushProperties appPushProperties;

    private IGtPush iGtPush;

    @Override
    public PlatformTypeEnum getType() {
        return PlatformTypeEnum.GETUI;
    }

    @Override
    void singlePush(PushMessage message, UserDeviceBean userDevice) {
        System.setProperty("gexin.rp.sdk.pushlist.needDetails", "true");
        IGtPush push = this.getIgtpush();

        ITemplate template = null;
        //因为iOS不支持个推的通知，只能采用透传的方式，所以设置无效
        if (TemplateTypeEnum.NOTIFCTION.equals(message.getType()) && !isIOS(userDevice)) {
            template = notificationTemplate(message);
        } else {
            template = transmissionTemplate(message);
        }

        SingleMessage singleMessage = new SingleMessage();
        singleMessage.setOffline(true);
        // 离线有效时间，单位为毫秒，可选
        singleMessage.setOfflineExpireTime(message.getPeriodTime() * 3600 * 1000);
        singleMessage.setData(template);
        // 可选，1为wifi，0为不限制网络环境。根据手机处于的网络情况，决定是否下发
        singleMessage.setPushNetWorkType(0);

        Target target = new Target();
        target.setAppId(this.appPushProperties.getId());
        if (StringUtils.isNotBlank(userDevice.getPushToken())) {
            target.setClientId(userDevice.getPushToken());
        } else {
            target.setAlias(userDevice.getUserId().toString());
        }

        IPushResult ret = push.pushMessageToSingle(singleMessage, target);

        String response = null;
        if (null != ret && ret.getResponse() != null) {
            response = JSON.toJSONString(ret.getResponse());
        }

        //推送完成回调
        callback(message, userDevice, response);
    }

    @Override
    void groupPush(PushMessage message, List<UserDeviceBean> userDevice) {
        for (UserDeviceBean userDeviceBean : userDevice) {
            singlePush(message, userDeviceBean);
        }
    }

    /**
     * 构建透传的消息模板
     * @param pushMessage 推送消息参数
     * @return 透传消息模板
     */
    private TransmissionTemplate transmissionTemplate(PushMessage pushMessage) {
        TransmissionTemplate template = new TransmissionTemplate();
        template.setAppId(this.appPushProperties.getId());
        template.setAppkey(this.appPushProperties.getKey());

        // 透传消息设置，1为强制启动应用，客户端接收到消息后就会立即启动应用；2为等待应用启动
        template.setTransmissionType(2);
        template.setTransmissionContent(pushMessage.getPayloadInfo().toJsonString());
        template.setAPNInfo(buildApnPayload(pushMessage));
        return template;
    }

    /**
     * 构建通知消息模板
     * @param pushMessage 推送消息参数
     * @return 通知消息模板
     */
    private NotificationTemplate notificationTemplate(PushMessage pushMessage) {
        NotificationTemplate template = new NotificationTemplate();
        template.setAppId(this.appPushProperties.getId());
        template.setAppkey(this.appPushProperties.getKey());
        template.setTransmissionType(1);
        template.setTransmissionContent(pushMessage.getPayloadInfo().toJsonString());
        template.setAPNInfo(buildApnPayload(pushMessage));

        Style0 style = new Style0();
        style.setTitle(pushMessage.getTitle());
        style.setText(pushMessage.getContent());
        style.setRing(!pushMessage.isMute());
        style.setVibrate(true);
        style.setClearable(true);

        template.setStyle(style);
        return template;
    }

    /**
     * 构建IOS消息体
     * @param pushMessage
     * @return
     */
    private APNPayload buildApnPayload(PushMessage pushMessage) {
        APNPayload payload = new APNPayload();
        payload.setAutoBadge("+1");
        if (TemplateTypeEnum.NOTIFCTION.equals(pushMessage.getType())) {
            payload.setContentAvailable(1);
        } else {
            payload.setContentAvailable(0);
        }
        if (pushMessage.isMute()) {
            payload.setSound(IOS_MUTE.getIosSound());
        } else {
            payload.setSound(pushMessage.getSound().getIosSound());
        }

        APNPayload.DictionaryAlertMsg dictionaryAlertMsg = new APNPayload.DictionaryAlertMsg();
        if (!PushMessageEnum.getMessageSilenceByType(pushMessage.getPayloadInfo().getType())) {
            dictionaryAlertMsg.setTitle(pushMessage.getTitle());
            dictionaryAlertMsg.setBody(pushMessage.getContent());
        } else {
            payload.setContentAvailable(0);
        }

        payload.setAlertMsg(dictionaryAlertMsg);
        payload.addCustomMsg("payload", pushMessage.getPayloadInfo().toJsonString());

        return payload;
    }

    /**
     * 获取IGtPush实例
     * @return
     */
    private IGtPush getIgtpush() {
        if (null == iGtPush) {
            iGtPush = new IGtPush(this.appPushProperties.getHost(), this.appPushProperties.getKey(), this.appPushProperties.getMasterSecret());
        }
        return iGtPush;
    }
}
