package com.bxm.localnews.mq.consume.push.platform.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.httpclient.service.HttpClientService;
import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.config.AppPushProperties;
import com.bxm.localnews.msg.vo.UserDeviceBean;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.MessageFormat;
import java.util.List;

/**
 * 华为推送平台
 * @author liujia
 */
@Component
public class HuaweiPushPlatform extends AbstractPushPlatform {

    @Resource
    private HttpClientService httpClientService;

    @Resource
    private AppPushProperties appPushProperties;

    private String accessToken;

    private long tokenExpiredTime = 0;

    private void execPush(PushMessage message, JSONArray deviceTokens, List<UserDeviceBean> spiltUserList) {
        if (tokenExpiredTime <= System.currentTimeMillis()) {
            refreshToken();
        }

        JSONObject param = new JSONObject();
        //定义需要打开的appPkgName
        param.put("appPkgName", appPushProperties.getAppPkgName());

        JSONObject action = new JSONObject();
        //类型3为打开APP，其他行为请参考接口文档设置
        action.put("type", 3);
        //消息点击动作参数
        action.put("param", param);

        JSONObject msg = new JSONObject();
        //3: 通知栏消息，异步透传消息请根据接口文档设置
        if (TemplateTypeEnum.NOTIFCTION.equals(message.getType())) {
            msg.put("type", 3);
        } else {
            msg.put("type", 1);
        }
        //消息点击动作
        msg.put("action", action);
        //通知栏消息body内容
        msg.put("body", JSONObject.toJSONString(message.getPayloadInfo()));

        //扩展信息，含BI消息统计，特定展示风格，消息折叠。
        JSONObject ext = new JSONObject();
        //设置消息标签，如果带了这个标签，会在回执中推送给CP用于检测某种类型消息的到达率和状态
        ext.put("biTag", "Trump");
        //自定义推送消息在通知栏的图标,value为一个公网可以访问的URL
        //ext.put("icon", "http://pic.qiantucdn.com/58pic/12/38/18/13758PIC4GV.jpg");

        //华为PUSH消息总结构体
        JSONObject hps = new JSONObject();
        hps.put("msg", msg);
        hps.put("ext", ext);

        JSONObject payload = new JSONObject();
        payload.put("hps", hps);

        String postBody;
        try {
            postBody = MessageFormat.format(
                    "access_token={0}&nsp_svc={1}&nsp_ts={2}&device_token_list={3}&payload={4}",
                    URLEncoder.encode(accessToken, "UTF-8"),
                    URLEncoder.encode("openpush.message.api.send", "UTF-8"),
                    URLEncoder.encode(String.valueOf(System.currentTimeMillis() / 1000), "UTF-8"),
                    URLEncoder.encode(deviceTokens.toString(), "UTF-8"),
                    URLEncoder.encode(payload.toString(), "UTF-8"));

            String postUrl = appPushProperties.getHuaweiApiUrl() + "?nsp_ctx=" +
                    URLEncoder.encode("{\"ver\":\"1\", \"appId\":\"" +
                            appPushProperties.getHuaweiAppId() + "\"}", "UTF-8");
            String response = httpClientService.doPost(postUrl, postBody);

            callback(message, spiltUserList, response);
            logger.debug("huawei push result:[{}]", response);
        } catch (UnsupportedEncodingException e) {
            logger.error(e.getMessage(), e);
        }
    }

    @Override
    void singlePush(PushMessage message, UserDeviceBean userDevice) {
        //目标设备Token
        JSONArray deviceTokens = new JSONArray();
        deviceTokens.add(userDevice.getPushToken());

        execPush(message, deviceTokens, ImmutableList.of(userDevice));
        callback(message, userDevice, null);
    }

    @Override
    void groupPush(PushMessage message, List<UserDeviceBean> userDeviceList) {
        JSONArray deviceTokens = new JSONArray();
        List<UserDeviceBean> spiltUserList = Lists.newArrayList();
        int index = 0;
        for (UserDeviceBean userDevice : userDeviceList) {
            deviceTokens.add(userDevice.getPushToken());
            spiltUserList.add(userDevice);

            //批量推送最多支持100条
            if (index != 0 && index % 100 == 0) {
                execPush(message, deviceTokens, spiltUserList);
                deviceTokens = new JSONArray();
                spiltUserList = Lists.newArrayList();
            }
            index++;
        }

        if (userDeviceList.size() % 100 != 0) {
            execPush(message, deviceTokens, spiltUserList);
        }
    }

    /**
     * 更新请求token
     */
    private void refreshToken() {
        String msgBody = null;
        try {
            msgBody = MessageFormat.format(
                    "grant_type=client_credentials&client_secret={0}&client_id={1}",
                    URLEncoder.encode(appPushProperties.getHuaweiClientSecret(), "UTF-8"), appPushProperties.getHuaweiAppId());
        } catch (UnsupportedEncodingException e) {
            logger.error(e.getMessage(), e);
        }
        String response = httpClientService.doPost(appPushProperties.getHuaweiAccessTokenUrl(), msgBody);
        JSONObject obj = JSONObject.parseObject(response);
        tokenExpiredTime = System.currentTimeMillis() + (obj.getLong("expires_in") - 5 * 60) * 1000;
        accessToken = obj.getString("access_token");
    }

    @Override
    public PlatformTypeEnum getType() {
        return PlatformTypeEnum.HUAWEI;
    }
}
