package com.bxm.localnews.msg.produce.impl;

import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.Producer;
import com.aliyun.openservices.ons.api.SendResult;
import com.bxm.localnews.mq.common.constant.AliMqMsgTagEnum;
import com.bxm.localnews.msg.config.AliyunMqProperties;
import com.bxm.localnews.msg.produce.AliMqService;
import com.bxm.newidea.cloud.constant.EnvConstant;
import com.bxm.newidea.cloud.ribbon.RibbonFilterContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

@Service
public class AliMqServiceImpl implements AliMqService {

    private static final Logger logger = LoggerFactory.getLogger(AliMqServiceImpl.class);

    private AliyunMqProperties aliyunMqProperties;

    /**
     * 存放所有ProducerBean的实现
     */
    private Map<String, Producer> producerBeanMap;

    private ApplicationContext applicationContext;

    @Resource
    private ObjectMapper objectMapper;

    @Autowired
    public AliMqServiceImpl(AliyunMqProperties aliyunMqProperties) {
        this.aliyunMqProperties = aliyunMqProperties;
    }

    @Override
    @Deprecated
    public <T> Message createMessage(String tag, T data) {
        Assert.hasText(tag, "tag is blank");
        Assert.notNull(data, "data is null");

        byte[] msgBody = null;
        try {
            msgBody = objectMapper.writeValueAsBytes(data);
        } catch (JsonProcessingException e) {
            logger.error("CREATE-MSG-ERR", e);
        }
        return new Message(aliyunMqProperties.getTopic(), tag, msgBody);
    }

    @Override
    public SendResult send(Message message) {
        Assert.notNull(message, "message is null");
        Assert.hasText(message.getTag(), "tag is blank");

        String tag = message.getTag();
        String producerName = AliMqMsgTagEnum.getProducerNameByTag(tag);

        Properties userProperties = message.getUserProperties();
        if (null == userProperties) {
            userProperties = new Properties();
        }
        String env = RibbonFilterContextHolder.getCurrentContext().get(EnvConstant.ENV_HEADER_KEY);
        if (StringUtils.isNotBlank(env)) {
            userProperties.setProperty(EnvConstant.ENV_HEADER_KEY, EnvConstant.PRE_ENV_VALUE);
            message.setUserProperties(userProperties);
        }

        return producerBeanMap.get(producerName).send(message);
    }

    @Override
    public <T> Message createMessage(AliMqMsgTagEnum tagEnum, T data) {
        Assert.notNull(tagEnum, "tag is blank");
        Assert.notNull(data, "data is null");

        byte[] msgBody = null;
        try {
            msgBody = objectMapper.writeValueAsBytes(data);
        } catch (JsonProcessingException e) {
            logger.error("CREATE-MSG-ERR", e);
        }

        return new Message(chooseTopic(tagEnum.getTag()), tagEnum.getTag(), msgBody);
    }

    /**
     * 获取tag对应的topic
     * @param tag
     * @return
     */
    private String chooseTopic(String tag) {
        String topicType = AliMqMsgTagEnum.getTopicTypeByTag(tag);
        switch (topicType) {
            case "sms":
                return aliyunMqProperties.getSmsTopic();
            case "push":
                return aliyunMqProperties.getPushTopic();
            case "news":
                return aliyunMqProperties.getNewsTopic();
            default:
                return aliyunMqProperties.getTopic();
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;

    }

    @Override
    public void onApplicationEvent(ContextRefreshedEvent contextRefreshedEvent) {
        Map<String, Producer> producerMap = applicationContext.getBeansOfType(Producer.class);
        producerBeanMap = new HashMap<>();
        producerMap.forEach((key, value) -> producerBeanMap.put(key, value));
    }

}
