package com.bxm.localnews.msg.push.platform.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.mybatis.utils.BatchHelper;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.domain.MessageMapper;
import com.bxm.localnews.msg.domain.PushMessageMapper;
import com.bxm.localnews.msg.domain.PushUserMapper;
import com.bxm.localnews.msg.push.platform.IPushPlatform;
import com.bxm.localnews.msg.service.MessageService;
import com.bxm.localnews.msg.vo.Message;
import com.bxm.localnews.msg.vo.PushUserBean;
import com.bxm.localnews.msg.vo.UserDeviceBean;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;
import java.util.List;

/**
 * 抽象的推送平台类
 * 处理推送相关的共同内容
 * @author liujia
 */
public abstract class AbstractPushPlatform implements IPushPlatform {

    Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private PushMessageMapper pushMessageMapper;

    @Autowired
    private MessageService messageService;

    @Override
    public void push(PushMessage message, List<UserDeviceBean> userDevice) {
        logger.debug("开始推送消息，message：[{}],userSize:[{}]", message, userDevice.size());
        if (userDevice.size() > 0) {
            if (userDevice.size() == 1) {
                singlePush(message, userDevice.get(0));
            } else {
                groupPush(message, userDevice);
            }
            changePushNum(message, userDevice);
        }
        logger.debug("推送消息结束，message：[{}],userSize:[{}]", message, userDevice.size());
    }

    /**
     * 推送完成后更新用户触达人数
     * 非默认（即指定单人推送或固定人数推送的情况）才需要进行人数的更新
     * @param message    消息信息
     * @param userDevice 用户列表
     */
    private void changePushNum(PushMessage message, List<UserDeviceBean> userDevice) {
        logger.debug("消息[{}]更新触达人数[{}]", message.getPayloadInfo().getMsgId(), userDevice.size());
        pushMessageMapper.updateNum(message.getPayloadInfo().getMsgId(), userDevice.size());
    }

    /**
     * 推送结果回调处理
     */
    void callback(PushMessage message, UserDeviceBean user, String response) {
        callback(message, ImmutableList.of(user), response);
    }

    void callback(PushMessage message, List<UserDeviceBean> users, String response) {
        new BatchHelper<PushUserMapper, UserDeviceBean>(PushUserMapper.class, users) {

            @Override
            protected int invoke(UserDeviceBean element) {
                PushUserBean entity = new PushUserBean();
                entity.setId(sequenceCreater.nextLongId());
                entity.setChecked(0);
                entity.setMessageId(message.getPayloadInfo().getMsgId());
                entity.setPlatform(element.getPlatform());
                entity.setUserId(element.getUserId());
                entity.setPushTime(new Date());
                entity.setPushResult(response);
                entity.setRegisterClient(StringUtils.isEmpty(element.getRegisterClient()) ?
                        "0" : element.getRegisterClient());
                return mapper.insert(entity);
            }
        };

        if (message.isPersistence()) {
            doPersistence(message, users);
        }
    }

    /**
     * 持久化用户通知表
     */
    private void doPersistence(PushMessage message, List<UserDeviceBean> users) {
        List<Message> messageList = Lists.newArrayList();
        for (UserDeviceBean userDeviceBean : users) {
            Message messageVo = new Message();
            messageVo.setTitle(message.getTitle());
            messageVo.setContent(message.getContent());
            messageVo.setUserId(userDeviceBean.getUserId());
            messageVo.setMsgType(String.valueOf(message.getPayloadInfo().getType()));
            messageVo.setStatus((byte) 0);
            messageVo.setLinkParam(JSON.toJSONString(message.getPayloadInfo()));
            messageVo.setAddTime(new Date());
            messageVo.setUpdateTime(new Date());
            messageList.add(messageVo);
        }

        MybatisBatchBuilder.create(MessageMapper.class, messageList).run((mapper, msg) -> {
            messageService.addUnreadMsg(message, msg.getUserId());
            return mapper.insertSelective(msg);
        });
    }

    /**
     * 推送消息到单独的用户
     * @param message    消息体
     * @param userDevice 用户设备信息
     */
    abstract void singlePush(PushMessage message, UserDeviceBean userDevice);

    /**
     * 推送消息到群组
     * @param message    消息体
     * @param userDevice 用户设备信息列表
     */
    abstract void groupPush(PushMessage message, List<UserDeviceBean> userDevice);

    /**
     * 判断用户设备是否iOS
     * @param userDevice 用户设备信息
     * @return true表示用户设备为iOS
     */
    boolean isIOS(UserDeviceBean userDevice) {
        if (null == userDevice) {
            return false;
        }

        //ios直接为true
        if (StringUtils.equals(PlatformEnum.IOS.getCode() + "", userDevice.getRegisterClient())) {
            return true;
        }

        //排除安卓
        if (StringUtils.equals(PlatformEnum.ANDROID.getCode() + "", userDevice.getRegisterClient())) {
            return false;
        }

        //存在历史错误数据，可能会注册客户端为3。根据设备模型补偿判断一次
        if (StringUtils.equals(PlatformEnum.WEB.getCode() + "", userDevice.getRegisterClient())) {
            return null != userDevice.getPhoneModel() && userDevice.getPhoneModel().contains("iPhone");
        }
        return false;
    }
}
