package com.bxm.localnews.mq.consume.push;

import java.util.List;
import java.util.Map;

import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.mq.consume.push.platform.IPushPlatform;
import com.bxm.localnews.mq.consume.push.receive.IReceiveUserIterator;
import com.bxm.localnews.mq.consume.push.receive.ReceiveUserFactory;
import com.bxm.localnews.msg.domain.PushMessageMapper;
import com.bxm.localnews.msg.vo.UserDeviceBean;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Maps;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

/**
 * 消息推送调度中心
 * 根据推送消息的接收者获取对应的接收用户列表
 * 将用户根据设备类型进行区分，根据具体的平台执行推送
 *
 * @author liujia
 */
@Component
public class PushMessageDispatcher implements ApplicationListener<ContextRefreshedEvent>, ApplicationContextAware {

    private static final Logger LOGGER = LoggerFactory.getLogger(PushMessageDispatcher.class);

    private final ReceiveUserFactory receiveUserManage;

    private final AsyncTaskExecutor asyncTaskExecutor;

    private final PushMessageMapper pushMessageMapper;

    private Map<PlatformTypeEnum, IPushPlatform> pushPlatformMap;

    private ApplicationContext applicationContext;

    private SequenceCreater sequenceCreater;

    @Autowired
    public PushMessageDispatcher(ReceiveUserFactory receiveUserManage, AsyncTaskExecutor asyncTaskExecutor,
                                 PushMessageMapper pushMessageMapper, SequenceCreater sequenceCreater) {
        this.receiveUserManage = receiveUserManage;
        this.asyncTaskExecutor = asyncTaskExecutor;
        this.pushMessageMapper = pushMessageMapper;
        this.sequenceCreater = sequenceCreater;
    }

    /**
     * 执行具体的推送
     * 1.根据用户推送范围获取到用户信息拉取的迭代器
     * 2.根据迭代器获取推送批次的用户
     * 3.异步进行推送动作
     *
     * @param message 推送消息参数
     */
    public void push(PushMessage message) {
        LOGGER.info("消息推送开始,message:{}", message);

        if (StringUtils.isEmpty(message.getPayloadInfo().getMsgId())) {
            Long msgId = sequenceCreater.nextLongId();
            message.getPayloadInfo().addExtend("msgId", msgId);
            message.getPayloadInfo().setMsgId(msgId);
        }
        
        PushReceiveScope receiveScope = message.getPushReceiveScope();
        //根据消息接收范围获取对应的用户信息
        IReceiveUserIterator iterator = receiveUserManage.get(receiveScope.getPushReceiverRule(), receiveScope.getRuleParam());

        List<PlatformTypeEnum> platformTypes = message.getPushReceiveScope().getPlatformTypes();

        while (iterator.hasNext()) {
            Map<PlatformTypeEnum, List<UserDeviceBean>> userMap = iterator.next();

            asyncTaskExecutor.execute(() -> userMap.forEach((type, users) -> {
                LOGGER.info("异步推送开始，推送平台：[{}],消息ID:[{}]推送的用户长度：[{}]", type,message.getPayloadInfo().getMsgId(), users.size());
                try {
                    if (!CollectionUtils.isEmpty(platformTypes)) {
                        for (PlatformTypeEnum platform : platformTypes) {
                            if (platform.equals(type)) {
                                pushPlatformMap.get(type).push(message, users);
                            }
                        }
                    } else {
                        pushPlatformMap.get(type).push(message, users);
                    }
                }catch (Exception e){
                    LOGGER.error("异步推送出现异常，message：[{}]，errorMessage：[{}]",message,e.getMessage());
                    LOGGER.error(e.getMessage(),e);
                }
                LOGGER.info("异步推送执行完成，mesasge：[{}]",message);
            }));

        }

        //完成推送后将消息更新为推送完成
        pushMessageMapper.updateStats(message.getPayloadInfo().getMsgId(), 2);

        LOGGER.info("消息推送结束,message:{}", message);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void onApplicationEvent(ContextRefreshedEvent event) {
        Map<String, IPushPlatform> pushPlatormBeanMap = applicationContext.getBeansOfType(IPushPlatform.class);
        if (pushPlatormBeanMap.size() == 0) {
            throw new IllegalArgumentException("push platform is emtpy");
        }
        pushPlatformMap = Maps.newHashMap();
        pushPlatormBeanMap.forEach((key, value) -> {
            pushPlatformMap.put(value.getType(), value);
        });

    }
}
