package com.bxm.localnews.mq.consume.push.platform.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.mybatis.utils.BatchHelper;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.consume.push.platform.IPushPlatform;
import com.bxm.localnews.mq.consume.service.MessageService;
import com.bxm.localnews.msg.domain.MessageMapper;
import com.bxm.localnews.msg.domain.PushMessageMapper;
import com.bxm.localnews.msg.domain.PushUserMapper;
import com.bxm.localnews.msg.vo.Message;
import com.bxm.localnews.msg.vo.PushUserBean;
import com.bxm.localnews.msg.vo.UserDeviceBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;
import java.util.List;

/**
 * 抽象的推送平台类
 * 处理推送相关的共同内容
 * @author liujia
 */
public abstract class AbstractPushPlatform implements IPushPlatform {

    Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private PushUserMapper pushUserMapper;

    @Autowired
    private PushMessageMapper pushMessageMapper;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private MessageService messageService;

    @Override
    public void push(PushMessage message, List<UserDeviceBean> userDevice) {
        logger.info("开始推送消息，message：[{}],userSize:[{}]", message, userDevice.size());
        if (userDevice.size() > 0) {
            if (userDevice.size() == 1) {
                singlePush(message, userDevice.get(0));
            } else {
                groupPush(message, userDevice);
            }
            changePushNum(message, userDevice);
        }
        logger.info("推送消息结束，message：[{}],userSize:[{}]", message, userDevice.size());
    }

    /**
     * 推送完成后更新用户触达人数
     * 非默认（即指定单人推送或固定人数推送的情况）才需要进行人数的更新
     * @param message    消息信息
     * @param userDevice 用户列表
     */
    private void changePushNum(PushMessage message, List<UserDeviceBean> userDevice) {
        logger.info("消息[{}]更新触达人数[{}]", message.getPayloadInfo().getMsgId(), userDevice.size());
        pushMessageMapper.updateNum(message.getPayloadInfo().getMsgId(), userDevice.size());
    }

    /**
     * 推送结果回调处理
     */
    void callback(PushMessage message, UserDeviceBean user, String response) {
        callback(message, ImmutableList.of(user), response);
    }

    void callback(PushMessage message, List<UserDeviceBean> users, String response) {
        new BatchHelper<PushUserMapper, UserDeviceBean>(PushUserMapper.class, users) {

            @Override
            protected int invoke(UserDeviceBean element) {
                PushUserBean entity = new PushUserBean();
                entity.setId(sequenceCreater.nextLongId());
                entity.setChecked(0);
                entity.setMessageId(message.getPayloadInfo().getMsgId());
                entity.setPlatform(element.getPlatform());
                entity.setUserId(element.getUserId());
                entity.setPushTime(new Date());
                entity.setPushResult(response);
                entity.setRegisterClient(StringUtils.isEmpty(element.getRegisterClient()) ?
                        "0" : element.getRegisterClient());
                return mapper.insert(entity);
            }
        };

        if (message.isPersistence()) {
            doPersistence(message, users);
        }
    }

    /**
     * 持久化用户通知表
     * @param message
     * @param users
     */
    private void doPersistence(PushMessage message, List<UserDeviceBean> users) {
        List<Message> messageList = Lists.newArrayList();
        for (UserDeviceBean userDeviceBean : users) {
            Message messageVo = new Message();
            messageVo.setTitle(message.getTitle());
            messageVo.setContent(message.getContent());
            messageVo.setUserId(userDeviceBean.getUserId());
            messageVo.setMsgType(String.valueOf(message.getPayloadInfo().getType()));
            messageVo.setStatus((byte) 0);
            messageVo.setLinkParam(JSON.toJSONString(message.getPayloadInfo()));
            messageList.add(messageVo);
        }
        new BatchHelper<MessageMapper, Message>(MessageMapper.class, messageList) {

            @Override
            protected int invoke(Message element) {
                KeyGenerator msgKey = messageService.getMsgKeyByType(element.getMsgType(), element.getUserId());
                logger.debug("当前用户[{}]的发送消息类型为[{}]，对应的redis-key[{}]", element.getUserId(),
                        element.getMsgType(), msgKey.gen());
                redisStringAdapter.increment(msgKey);
                int success = mapper.insertSelective(element);
                //这里添加未读消息时间必须在消息入库之后再进行查询-----保证未读消息获取到最新的条数
                messageService.addUnReadUserEvent(element.getUserId(), element.getMsgType());
                return success;
            }
        };
    }

    /**
     * 推送消息到单独的用户
     * @param message    消息体
     * @param userDevice 用户设备信息
     */
    abstract void singlePush(PushMessage message, UserDeviceBean userDevice);

    /**
     * 推送消息到群组
     * @param message    消息体
     * @param userDevice 用户设备信息列表
     */
    abstract void groupPush(PushMessage message, List<UserDeviceBean> userDevice);

    /**
     * 判断用户设备是否iOS
     * @param userDevice 用户设备信息
     * @return
     */
    boolean isIOS(UserDeviceBean userDevice) {
        return null != userDevice && ((null != userDevice.getRegisterClient() && userDevice.getRegisterClient().equals("2"))
                || (null != userDevice.getPhoneModel() && userDevice.getPhoneModel().indexOf("iPhone") != -1));
    }
}
