package com.bxm.localnews.mq.consume.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.MPage;
import com.bxm.localnews.mq.common.RedisConfig;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.config.MessageTypeProperties;
import com.bxm.localnews.mq.consume.service.MessageService;
import com.bxm.localnews.mq.event.service.UserEventService;
import com.bxm.localnews.msg.domain.MessageMapper;
import com.bxm.localnews.msg.dto.MessageTypeDTO;
import com.bxm.localnews.msg.integration.AdvertIntegrationService;
import com.bxm.localnews.msg.integration.AppVersionIntegrationService;
import com.bxm.localnews.msg.integration.PushTemplateMessageIntegrationService;
import com.bxm.localnews.msg.param.MessageListParam;
import com.bxm.localnews.msg.vo.Message;
import com.bxm.localnews.msg.vo.MessageType;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.newidea.component.tools.DateUtils.addField;

@Service
@Transactional(rollbackFor = Exception.class)
@RefreshScope
@Slf4j
public class MessageServiceImpl implements MessageService {

    @Resource
    private MessageMapper messageMapper;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private AdvertIntegrationService advertIntegrationService;

    @Resource
    private MessageTypeProperties messageTypeProperties;

    @Resource
    private AppVersionIntegrationService appVersionIntegrationService;

    @Resource
    private UserEventService userEventService;

    @Resource
    private PushTemplateMessageIntegrationService pushTemplateMessageIntegrationService;

    @Override
    public int getUnReadMsg(Long userId, Integer platform) {
        int num = getNewMsg(userId);
        int unreadnum;
        if (platform == 5) {
            unreadnum = redisStringAdapter.getInt(getKeyByType("BALANCE", userId));
        } else {
            unreadnum = redisStringAdapter.getInt(getKeyByType("INVITE", userId));
        }
        log.debug("当前用户[{}]未读消息[{}]--------对应平台需去除的未读消息条数[{}]", userId, num, unreadnum);
        return num - unreadnum;
    }

    @Override
    public int getNewMsg(Long userId) {
        return this.messageMapper.getNewMsg(userId);
    }

    @Override
    public List<Message> queryNoticeList(Long userId, MPage page) {
        return this.messageMapper.queryNoticeList(userId, page);
    }

    @Override
    public Message noticeDetail(Integer id) {
        return this.messageMapper.selectByPrimaryKey(id);
    }

    @Override
    public PageWarper<Message> listMessage(MessageListParam messageListParam, BasicParam basicParam) {
        if ("INTERACTION".equals(messageListParam.getMessageType())) {
            return new PageWarper<>(Lists.newArrayList());
        }
        messageListParam = constructionParam(messageListParam);
        //小程序的系统通知只查询任务类型
        if ("SYSTEM".equals(messageListParam.getMessageType()) && basicParam.getPlatform() == 5) {
            messageListParam.setMsgTypeList(Lists.newArrayList(String.valueOf(PushMessageEnum.TASK_COMPLETED.getType())));
        }
        PageWarper<Message> messagePageWarper = new PageWarper<>(messageMapper.listMessage(messageListParam));

        String messageType = messageListParam.getMessageType();
        //设置用户图标
        messagePageWarper.getList().forEach(message -> {
            message.setReadStatus(message.getStatus());
            message.setLastTime(changeTime(message.getAddTime()));

            if (StringUtils.isNotEmpty(messageType) && ("COMMENT".equals(messageType) || "LIKE".equals(messageType)
                    || "SHARE".equals(messageType) || "INVITE".equals(messageType))) {
                JSONObject extend = JSON.parseObject(message.getLinkParam());
                message.setNickname(extend.getJSONObject("extend").getString("nickname"));
                message.setAction(extend.getJSONObject("extend").getString("action"));
                message.setIcon(extend.getJSONObject("extend").getString("icon"));
                message.setExtendContent(extend.getJSONObject("extend").getString("extendContent"));
                message.setExtendUrl(extend.getJSONObject("extend").getString("extendUrl"));
            }
        });
        return messagePageWarper;
    }

    @Override
    public MessageTypeDTO getMessageType(String areaCode, Long userId, BasicParam basicParam) {
        MessageTypeDTO messageTypeDTO = new MessageTypeDTO();
        messageTypeDTO.setAdvertDTOS(advertIntegrationService.getAdvertByType("10", areaCode, userId));
        List<MessageType> messageTypeList = getMessageTypes(userId);
        if (appVersionIntegrationService.getPublishState(basicParam)) {
            messageTypeList = messageTypeList.stream().filter(messageType ->
                    !"BALANCE".equals(messageType.getMessageType())).collect(Collectors.toList());
        }

        //备注：平台：5 移除余额变动通知和系统通知 -------平台：1和2 移除邀请通知
        if (basicParam.getPlatform() == 5) {
            messageTypeList = messageTypeList.stream().filter(messageType ->
                    !("BALANCE".equals(messageType.getMessageType()))).collect(Collectors.toList());
        } else if (basicParam.getPlatform() == 1 || basicParam.getPlatform() == 2) {
            messageTypeList = messageTypeList.stream().filter(messageType ->
                    !"INVITE".equals(messageType.getMessageType())).collect(Collectors.toList());
        }

        messageTypeDTO.setMessageTypes(messageTypeList);
        messageTypeDTO.getMessageTypes().forEach(messageType -> {
            MessageListParam param = new MessageListParam();
            param.setAreaCode(areaCode);
            param.setUserId(userId);
            param.setMessageType(messageType.getMessageType());
            constructionParam(param);
            //FIXME 需要修改,避免多次查询，消息库需要分表
            Message message = messageMapper.selectLastOneByType(param.getUserId(), param.getMsgTypeList());
            if (message != null) {
                messageType.setDescription(message.getContent());
                if ("COMMENT".equals(messageType.getMessageType())
                        || "LIKE".equals(messageType.getMessageType())
                        || "SHARE".equals(messageType.getMessageType())) {
                    JSONObject extend = JSON.parseObject(message.getLinkParam());
                    message.setNickname(extend.getJSONObject("extend").getString("nickname"));
                    message.setAction(extend.getJSONObject("extend").getString("action"));
                    messageType.setDescription(message.getNickname() + message.getAction());
                }
                messageType.setLastTime(changeTime(message.getAddTime()));
            }
        });

        return messageTypeDTO;
    }

    @Override
    public KeyGenerator getMsgKeyByType(String messageType, Long userId) {
        KeyGenerator keyGenerator = getKeyByType(messageType, userId);
        if (keyGenerator != null) {
            return keyGenerator;
        }
        return getKeyByType(getMessageType(messageType), userId);
    }

    @Override
    public void updateMessageByType(Long userId, String messageType) {
        MessageListParam messageListParam = new MessageListParam();
        messageListParam.setUserId(userId);
        messageListParam.setMessageType(messageType);
        constructionParam(messageListParam);
        this.messageMapper.updateMessageByType(messageListParam);
    }

    @Override
    public Boolean addMessage(PushMessage message, Long userId) {
        Message messageVo = new Message();
        messageVo.setTitle(message.getTitle());
        messageVo.setContent(message.getContent());
        messageVo.setUserId(userId);
        messageVo.setMsgType(String.valueOf(message.getPayloadInfo().getType()));
        messageVo.setStatus((byte) 0);
        messageVo.setLinkParam(JSON.toJSONString(message.getPayloadInfo()));

        int addMessage = this.messageMapper.insertSelective(messageVo);

        //对应类型的未读消息增加

        log.debug("用户[{}]增加消息类型的[{}]: 消息参数为：[{}]", userId, messageVo.getMsgType(), JSON.toJSONString(messageVo));

        redisStringAdapter.increment(getMsgKeyByType(messageVo.getMsgType(), userId));
        //添加用户事件

        addUnReadUserEvent(userId, messageVo.getMsgType());

        //推送模板消息
        pushTemplateMessage(message);
        return addMessage > 0;
    }

    @Override
    public Boolean addUnReadUserEvent(Long userId, String msgType) {
        log.debug("用户[{}]添加未读消息事件", userId);

        if ("BALANCE".equals(getMessageType(msgType))) {
            userEventService.add(buildPushPayloadInfo(userId));
        } else if ("INVITE".equals(getMessageType(msgType))) {
            userEventService.add(buildAppletPushPayloadInfo(userId));
        } else {
            userEventService.add(buildPushPayloadInfo(userId));
            userEventService.add(buildAppletPushPayloadInfo(userId));
        }

        return true;
    }

    private PushPayloadInfo buildPushPayloadInfo(Long userId) {
        return PushPayloadInfo.build(PushMessageEnum.USER_UN_READ_MSG)
                .addExtend("userId", userId).addExtend("current", getUnReadMsg(userId, 0));
    }

    private PushPayloadInfo buildAppletPushPayloadInfo(Long userId) {
        return PushPayloadInfo.build(PushMessageEnum.APPLET_USER_UN_READ_MSG)
                .addExtend("userId", userId).addExtend("current", getUnReadMsg(userId, 5));
    }

    @Override
    public List<Message> getMessageListByType(List<String> typeList) {
        if (CollectionUtils.isEmpty(typeList)) {
            return null;
        }

        List<String> msgTypeList = new ArrayList<>();
        typeList.forEach(e -> {
            List<String> partMsgTypeList = messageTypeProperties.getMsgTypeMap().get(e);
            if (!CollectionUtils.isEmpty(partMsgTypeList)) {
                msgTypeList.addAll(partMsgTypeList);
            }
        });
        if (CollectionUtils.isEmpty(msgTypeList)) {
            return null;
        }

        Date now = new Date();
        Date beforeNow = DateUtils.addField(now, Calendar.MINUTE, -30);
        String startTime = DateUtils.formatDateTime(beforeNow);
        String endTime = DateUtils.formatDateTime(now);
        return messageMapper.getMessageListByType(msgTypeList, startTime, endTime);
    }

    @Override
    public void pushTemplateMessage(PushMessage pushMessage) {
        Integer type = pushMessage.getPayloadInfo().getType();
        if (messageTypeProperties.getTemplateList().contains(type)) {
            pushTemplateMessageIntegrationService.pushTemplateMessage(pushMessage, null);
        }
    }

    /**
     * 变更时间
     * @param time
     * @return
     */
    private String changeTime(Date time) {
        Date today = new Date();
        String changeTime = DateUtils.formatAtWill(time, DateUtils.DATE_FORMAT);

        if (org.apache.commons.lang3.time.DateUtils.isSameDay(time, today)) {
            return "今天" + DateUtils.PATTERN_HOUR_MINUTE_FORMAT.get().format(time);
        }

        if (org.apache.commons.lang3.time.DateUtils.isSameDay(addField(time, Calendar.DATE, 1), today)) {
            return "昨天";
        }

        int todayYear = today.getYear() + 1900;
        if (!StringUtils.split(changeTime, "-")[0].equals(String.valueOf(todayYear))) {
            return changeTime.replaceAll("-", "/");
        }

        if (StringUtils.split(changeTime, "-")[0].equals(String.valueOf(todayYear))) {
            return StringUtils.split(changeTime, "-")[1] + "/" +
                    StringUtils.split(changeTime, "-")[2];
        }

        return null;
    }

    /**
     * 获取图片和名称
     * @return
     */
    private List<MessageType> getMessageTypes(Long userId) {
        List<String> typeName = messageTypeProperties.getTypeName();
        List<String> typeImg = messageTypeProperties.getTypeImg();
        List<String> typeTitle = messageTypeProperties.getTypeTitle();
        List<MessageType> list = Lists.newArrayListWithCapacity(typeName.size());
        for (int i = 0; i < typeName.size(); i++) {
            MessageType messageType = new MessageType();
            messageType.setMessageType(typeName.get(i));
            messageType.setImg(typeImg.get(i));
            messageType.setTitle(typeTitle.get(i));
            messageType.setUnreadNum(redisStringAdapter.getInt(getKeyByType(typeName.get(i), userId)));
            list.add(messageType);
        }


        return list;
    }

    /**
     * 获取消息类型的key
     * @return
     */
    private KeyGenerator getKeyByType(String type, Long userId) {
        if ("SYSTEM".equals(type)) {
            return RedisConfig.SYSTEM_MSG.copy().appendKey(userId).appendKey("unread_num");
        } else if ("NEWS".equals(type)) {
            return RedisConfig.NEWS_MSG.copy().appendKey(userId).appendKey("unread_num");
        } else if ("BALANCE".equals(type)) {
            return RedisConfig.BALANCE_MSG.copy().appendKey(userId).appendKey("unread_num");
        } else if ("COMMENT".equals(type)) {
            return RedisConfig.COMMENT_MSG.copy().appendKey(userId).appendKey("unread_num");
        } else if ("LIKE".equals(type)) {
            return RedisConfig.LIKE_MSG.copy().appendKey(userId).appendKey("unread_num");
        } else if ("SHARE".equals(type)) {
            return RedisConfig.SHARE_MSG.copy().appendKey(userId).appendKey("unread_num");
        } else if ("INVITE".equals(type)) {
            return RedisConfig.INVITE_MSG.copy().appendKey(userId).appendKey("unread_num");
        } else if ("TASK".equals(type)) {
            return RedisConfig.TASK_MSG.copy().appendKey(userId).appendKey("unread_num");
        }
        log.debug("查询消息类型[{}]--------未匹配到redisKey", type);
        return null;
    }

    /**
     * 获取对应消息类型的消息标题
     * @param msgType
     * @return
     */
    private String getMessageType(String msgType) {

        Iterator<Map.Entry<String, List<String>>> iterator = messageTypeProperties.getMsgTypeMap().entrySet().iterator();
        log.debug("msgType:{},messTypeMap：{}", msgType, JSONObject.toJSONString(messageTypeProperties.getMsgTypeMap()));
        while (iterator.hasNext()) {
            Map.Entry<String, List<String>> entry = iterator.next();
            for (String str : entry.getValue()) {
                if (msgType.equals(str)) {
                    return entry.getKey();
                }
            }
        }

        return null;
    }

    /**
     * 构造参数信息
     * @param messageListParam
     * @return
     */
    private MessageListParam constructionParam(MessageListParam messageListParam) {
        //备注：小程序版本1.0.0和万事通2.3.0版本互动消息变成评论通知和点赞通知
        if (StringUtils.isNotEmpty(messageListParam.getMessageType())
                && "INTERACTION".equals(messageListParam.getMessageType())) {
            List<String> msgTypeList = messageTypeProperties.getMsgTypeMap().get("COMMENT");
            msgTypeList.addAll(messageTypeProperties.getMsgTypeMap().get("LIKE"));
            messageListParam.setMsgTypeList(msgTypeList);
        } else {
            messageListParam.setMsgTypeList(messageTypeProperties.getMsgTypeMap().get(messageListParam.getMessageType()));
        }

        return messageListParam;
    }
}
