package com.bxm.localnews.msg.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.util.DateFormatUtils;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.mq.common.constant.*;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.msg.config.InteractMessageProperties;
import com.bxm.localnews.msg.config.MessageTypeProperties;
import com.bxm.localnews.msg.constant.RedisConfig;
import com.bxm.localnews.msg.domain.MessageMapper;
import com.bxm.localnews.msg.dto.InteractMessageDTO;
import com.bxm.localnews.msg.dto.MessageType;
import com.bxm.localnews.msg.dto.MessageTypeDTO;
import com.bxm.localnews.msg.dto.MessageTypeNewDTO;
import com.bxm.localnews.msg.event.UserEventService;
import com.bxm.localnews.msg.integration.AdvertIntegrationService;
import com.bxm.localnews.msg.integration.AppVersionIntegrationService;
import com.bxm.localnews.msg.integration.PushTemplateMessageIntegrationService;
import com.bxm.localnews.msg.param.MessageListParam;
import com.bxm.localnews.msg.param.MsgCenterNewParam;
import com.bxm.localnews.msg.service.UserMessageService;
import com.bxm.localnews.msg.service.filter.MessageFilterStrategyManage;
import com.bxm.localnews.msg.utils.IntervalPeriodUtils;
import com.bxm.localnews.msg.vo.Tuple;
import com.bxm.localnews.msg.vo.UserMessageBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.joda.time.DateTimeComparator;
import org.springframework.beans.BeanUtils;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.parseObject;
import static com.bxm.localnews.mq.common.constant.MessageTypeEnum.*;
import static com.bxm.localnews.msg.constant.RedisConfig.*;
import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.isNotEmpty;

@Service
@Transactional(rollbackFor = Exception.class)
@RefreshScope
@Slf4j
public class UserMessageServiceImpl implements UserMessageService {

    private static final String NICKNAME = "nickname";

    private static final String EXTEND = "extend";

    private static final String ACTION = "action";

    private static final String ICON = "icon";

    private static final String EXTEND_CONTENT = "extendContent";

    private static final String EXTEND_URL = "extendUrl";

    private static final String HEAD_IMG = "headImg";

    private static final String CUSTOM_TYPE = "customType";

    private static final String REPLY_ID = "replyId";

    @Resource
    private MessageMapper messageMapper;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private AdvertIntegrationService advertIntegrationService;

    @Resource
    private MessageTypeProperties messageTypeProperties;

    @Resource
    private AppVersionIntegrationService appVersionIntegrationService;

    @Resource
    private UserEventService userEventService;

    private DateTimeComparator dateTimeComparator = DateTimeComparator.getInstance();

    @Resource
    private PushTemplateMessageIntegrationService pushTemplateMessageIntegrationService;

    @Resource
    private SequenceCreater sequenceCreater;

    @Resource
    private InteractMessageProperties interactMessageProperties;

    @Resource
    private MessageFilterStrategyManage messageFilterStrategyManage;

    @Override
    public int getUnReadMsg(Long userId, Integer platform, String curVer) {
        Map<String, Long> unreadNumMap = getUnreadMsgNumMap(userId);


        if (platform == PlatformEnum.APPLET.getCode()) {
            unreadNumMap.remove(MessageTypeEnum.BALANCE.name());
        } else {
            unreadNumMap.remove(MessageTypeEnum.INVITE.name());
        }

        Long unread = unreadNumMap.values().stream().mapToLong(v -> v).sum();

        if (log.isDebugEnabled()) {
            log.debug("{}在平台{},客户端版本：{},未读消息总数为：{}", userId, platform, curVer, unread);
        }

        return unread.intValue();
    }

    @Override
    public PageWarper<UserMessageBean> listMessage(MessageListParam messageListParam, BasicParam basicParam) {
        if (MessageTypeEnum.INTERACTION.name().equals(messageListParam.getMessageType())) {
            return new PageWarper<>(Lists.newArrayList());
        }

        constructionParam(messageListParam);

        if (CollectionUtils.isEmpty(messageListParam.getMsgTypeList())) {
            return new PageWarper<>(Lists.newArrayList());
        }

        if (log.isDebugEnabled()) {
            log.debug("用户消息列表，查询参数：{}", JSON.toJSONString(messageListParam));
        }
        PageWarper<UserMessageBean> msgPage = new PageWarper<>(messageMapper.listMessage(messageListParam));

        String messageType = messageListParam.getMessageType();
        //设置用户图标
        msgPage.getList().forEach(message -> {
            message.setReadStatus(message.getStatus());
            message.setLastTime(DateFormatUtils.format(message.getAddTime()));

            if (NumberUtils.isDigits(message.getMsgType())) {
                MessageBehaviorEnum behavior = PushMessageEnum.getMessageBehaviorByType(Integer.parseInt(message.getMsgType()));
                if (null != behavior) {
                    message.setHasDetail(MessageBehaviorEnum.JUMP.equals(behavior));
                }
            }

            //是否为互动消息
            if (messageTypeProperties.isInteraction(messageType)) {

                JSONObject linkParam = parseObject(message.getLinkParam());
                JSONObject extend = linkParam.getJSONObject(EXTEND);

                message.setNickname(extend.getString(NICKNAME));
                message.setAction(extend.getString(ACTION));
                message.setIcon(extend.getString(ICON));
                message.setExtendContent(extend.getString(EXTEND_CONTENT));
                message.setExtendUrl(extend.getString(EXTEND_URL));
                message.setReplyId(extend.getLong(REPLY_ID));

                message = (UserMessageBean) messageFilterStrategyManage.excute(message,messageType,MessageFilterEventTypeEnum.COMMENT_REPLY_DELETE_EVENT);
            }
        });
        return msgPage;
    }

    @Override
    public MessageTypeDTO getMessageType(String areaCode, Long userId, BasicParam basicParam) {
        MessageTypeDTO msgDto = new MessageTypeDTO();

        List<String> skipTypes = Lists.newArrayList();

        //提审时不显示余额 备注：平台：5 移除余额变动通知和系统通知 -------
        if (basicParam.getPlatform() == PlatformEnum.APPLET.getCode()
                || Boolean.TRUE.equals(appVersionIntegrationService.getPublishState(basicParam))) {
            skipTypes.add(MessageTypeEnum.BALANCE.name());
        }

        //平台：1和2 移除邀请通知
        if (basicParam.getPlatform() == PlatformEnum.ANDROID.getCode()
                || basicParam.getPlatform() == PlatformEnum.IOS.getCode()) {
            skipTypes.add(MessageTypeEnum.INVITE.name());
        }

        //设置用户的最后未读消息
        setMessageList(msgDto, userId, skipTypes, basicParam);

        //附加广告信息
        msgDto.setAdvertDTOS(advertIntegrationService.getAdvertByType("10", areaCode, userId));

        return msgDto;
    }

    /**
     * 获取消息中心显示的类型、标题、图标、未读数量。用于显示消息中心的列表
     *
     * @param msgDto    消息传输对象
     * @param userId    用户ID
     * @param skipTypes 不显示的消息类型
     */
    private void setMessageList(MessageTypeDTO msgDto, Long userId, List<String> skipTypes, BasicParam basicParam) {
        Map<String, Long> unreadNumMap = getUnreadMsgNumMap(userId);
        Map<String, MessageType> lastMsgMap = getLastMsgMap(userId);

        List<MessageType> msgList = lastMsgMap.values()
                .stream()
                .sorted((pre, next) ->
                        dateTimeComparator.compare(next.getSourceLastTime(), pre.getSourceLastTime()))
                .collect(Collectors.toList());

        List<MessageType> afterFilter = Lists.newArrayList();

        for (MessageType msgType : msgList) {
            String type = msgType.getMessageType();
            //跳过不在特定平台显示的消息
            if (null == type || skipTypes.contains(msgType.getMessageType())) {
                continue;
            }

            //三种特殊的消息类型处理
            if (StringUtils.equalsAny(type, LIKE.name(), COMMENT.name(), ADD_FUNS.name())) {
                continue;
            }

            // 数字类型的为私聊消息，不进行处理
            if (NumberUtils.isDigits(type)) {
                continue;
            }

            //处理日期格式
            if (null != msgType.getSourceLastTime()) {
                msgType.setLastTime(DateFormatUtils.format(msgType.getSourceLastTime()));
            }

            msgType.setUnreadNum(getUnreadIntVal(unreadNumMap, type));

            MessageType configTypeInfo = messageTypeProperties.getMessageInfo(type);

            if (log.isDebugEnabled()) {
                log.debug("match msg type:{}", configTypeInfo);
            }

            if (null != configTypeInfo) {
                msgType.setTitle(configTypeInfo.getTitle());
                msgType.setImg(configTypeInfo.getImg());
            } else {
                //未匹配的类型不再显示
                continue;
            }

            afterFilter.add(msgType);
        }
        msgDto.setMessageTypes(afterFilter);
        fillNull(msgDto, unreadNumMap);
    }


    private int getUnreadIntVal(Map<String, Long> unreadNumMap, String type) {
        Long unreadNum = unreadNumMap.get(type);
        if (unreadNum == null) {
            return 0;
        }
        return unreadNum.intValue();
    }

    /**
     * 点赞、评论、新增粉丝在某一个版本调整出了消息栏目到单独的顶部区域
     *
     * @param msgDto 消息栏目信息完整信息
     */
    private void fillNull(MessageTypeDTO msgDto, Map<String, Long> unreadNumMap) {
        MessageType likeType = new MessageType();
        likeType.setMessageType(LIKE.name());
        likeType.setTitle("点赞");
        likeType.setUnreadNum(getUnreadIntVal(unreadNumMap, LIKE.name()));
        msgDto.setLikeType(likeType);

        MessageType commentType = new MessageType();
        commentType.setMessageType(COMMENT.name());
        commentType.setTitle("评论");
        commentType.setUnreadNum(getUnreadIntVal(unreadNumMap, COMMENT.name()));
        msgDto.setCommentType(commentType);

        MessageType followType = new MessageType();
        followType.setMessageType(ADD_FUNS.name());
        followType.setTitle("新增粉丝");
        followType.setUnreadNum(getUnreadIntVal(unreadNumMap, ADD_FUNS.name()));
        msgDto.setFollowType(followType);

        if (msgDto.getMessageTypes() == null) {
            msgDto.setMessageTypes(Lists.newArrayList());
        }

        // 进行空值的填充
        for (String type : messageTypeProperties.getMessageTypes()) {
            if (StringUtils.equalsAny(type, LIKE.name(), COMMENT.name(), ADD_FUNS.name())) {
                continue;
            }
            if (msgDto.getMessageTypes().stream().anyMatch(item -> type.equals(item.getMessageType()))) {
                continue;
            }
            MessageType innerType = new MessageType();
            MessageType configTypeInfo = messageTypeProperties.getMessageInfo(type);
            if (null != configTypeInfo && null != configTypeInfo.getTitle()) {
                innerType.setMessageType(type);
                innerType.setTitle(configTypeInfo.getTitle());
                innerType.setImg(configTypeInfo.getImg());

                msgDto.getMessageTypes().add(innerType);
            }
        }
    }

    private KeyGenerator getUnreadCacheKey(Long userId) {
        return UN_READ_MSG.copy().appendKey(userId.toString());
    }

    private KeyGenerator getLastMsgCacheKey(Long userId) {
        return LAST_MSG.copy().appendKey(userId.toString());
    }

    /**
     * 获取用户的各种类型的未读消息数量
     *
     * @param userId 用户ID
     * @return 未读消息map，key为消息类型，value为对应消息类型的未读数量
     */
    private Map<String, Long> getUnreadMsgNumMap(Long userId) {
        KeyGenerator key = getUnreadCacheKey(userId);
        Map<String, Long> result = redisHashMapAdapter.entries(key, Long.class);

        //如果不存在用户缓存，则检索一次数据库
        if (null == result || result.isEmpty()) {
            result = loadHistoryUnread(userId);
            redisHashMapAdapter.putAll(key, result);
        }

        // V3.4.0 未读消息中移除私聊相关的数量
        for (Map.Entry<String, Long> unreadEntry : result.entrySet()) {
            String subKey = unreadEntry.getKey();
            if (NumberUtils.isDigits(subKey)
                    || PushMessageEnum.IM_MESSAGE.name().equalsIgnoreCase(subKey)) {
                redisHashMapAdapter.remove(key, subKey);
            }
        }

        return result;
    }

    /**
     * 从数据库加载各类型的历史未读数据
     *
     * @param userId 用户ID
     * @return 各消息类型的未读数量
     */
    private Map<String, Long> loadHistoryUnread(Long userId) {
        //查询各种推送消息的未读数量
        List<Tuple> unreadTypes = messageMapper.getUnreadMap(userId);

        Map<String, Long> result = Maps.newHashMap();

        for (Tuple tuple : unreadTypes) {
            //将推送消息的类型转换为消息中心显示的类型
            String msgType = messageTypeProperties.convertType(tuple.getKey());
            Long val = result.get(msgType);
            val = (val == null ? 0 : val) + tuple.getValue();

            result.put(msgType, val);
        }

        return result;
    }

    private Map<String, MessageType> getLastMsgMap(Long userId) {
        KeyGenerator key = getLastMsgCacheKey(userId);
        Map<String, MessageType> result;
        if (Boolean.TRUE.equals(redisHashMapAdapter.hasKey(key))) {
            result = redisHashMapAdapter.entries(key, MessageType.class);

            // 处理历史用户的错误消息线上，清理不必要的缓存
            result.forEach((type, msg) -> {
                if (NumberUtils.isDigits(type)) {
                    redisHashMapAdapter.remove(key, type);
                } else if (messageTypeProperties.getMessageInfo(type).getTitle() == null) {
                    redisHashMapAdapter.remove(key, type);
                }
            });
        } else {
            result = loadLastMsgHistory(userId);
            redisHashMapAdapter.putAll(key, result);
        }

        return result;
    }

    private Map<String, MessageType> loadLastMsgHistory(Long userId) {
        Map<String, MessageType> result = Maps.newHashMap();

        List<UserMessageBean> lastMsgType = messageMapper.getLastMsgType(userId);
        List<Long> lastIdList = lastMsgType.stream().map(UserMessageBean::getId).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(lastIdList)) {
            return result;
        }

        List<UserMessageBean> lastMessages = messageMapper.getLastMsg(userId, lastIdList);

        for (UserMessageBean msg : lastMessages) {
            String msgType = messageTypeProperties.convertType(msg.getMsgType());
            MessageType messageType = build(msg, msgType);

            result.put(msgType, messageType);
        }

        return result;
    }

    @Override
    public void addUnreadMsg(PushMessage message, Long userId) {
        int pushMessageType = message.getPayloadInfo().getType();
        String msgType = messageTypeProperties.convertType(pushMessageType);


        //V3.4.0 私聊消息不加入到未读消息总数
        if (PushMessageEnum.IM_MESSAGE.name().equalsIgnoreCase(msgType) ||
                msgType.equalsIgnoreCase(PushMessageEnum.IM_MESSAGE.getType() + "s")) {
            return;
        }

        //添加对应类型的未读消息,数据迁移原因，初次加载时，尝试加载历史数据
        KeyGenerator key = getUnreadCacheKey(userId);

        if (Boolean.FALSE.equals(redisHashMapAdapter.hasKey(key))) {
            redisHashMapAdapter.putAll(key, loadHistoryUnread(userId));
            redisHashMapAdapter.expire(key, 7 * 24 * 3600L);
        }

        redisHashMapAdapter.increment(key, msgType, 1);

        //添加对应类型的最后阅读消息
        addMessageTypeLastMsg(message, userId, msgType);

        //小程序模板消息推送
        //pushTemplateMessage(message, userId);

        log.info("添加未读消息，用户id：{}, 未读消息类型：{}", userId, pushMessageType);
        // 如果消息是点赞或者评论
        if (PushMessageEnum.BELONG_POST_REPLY.getType() == pushMessageType
                || PushMessageEnum.BELONG_POST_LIKE.getType() == pushMessageType
        ) {
            //添加用户互动消息未读数量变化事件
            InteractMessageDTO interactMessageDTO = getUnReadInteractMessage(userId);
            addInteractUserEvent(userId, interactMessageDTO);
        }


        //添加消息未读的事件
        addUnReadUserEvent(userId, msgType);

        //如果是互动消息，则计入到定期的互动消息中，统一进行推送
        if (messageTypeProperties.isInteraction(msgType)) {
            addInteractionCache(message, userId, msgType);
        }
    }

    /**
     * 添加互动消息未读事件
     *
     * @param userId             用户id
     * @param interactMessageDTO 未读互动消息信息
     */
    private void addInteractUserEvent(Long userId, InteractMessageDTO interactMessageDTO) {
        PushPayloadInfo interactMessageInfo = PushPayloadInfo.build().setType(UserEventEnum.UN_READ_INTERACT_MSG.type)
                .addExtend("messageCount", interactMessageDTO.getMessageCount()).addExtend("headImageUrl", interactMessageDTO.getHeadImageUrl())
                .addExtend("latestMessageType", interactMessageDTO.getLatestMessageType())
                .addExtend("jumpUrl", interactMessageDTO.getJumpUrl())
                .addExtend("userId", userId);
        log.info("添加互动消息未读事件：userId:{},消息内容：{}", userId, JSON.toJSONString(interactMessageDTO));
        //添加对应的事件
        userEventService.add(interactMessageInfo);
    }

    @Override
    public void addLastMsg(PushMessage message) {
        Preconditions.checkArgument(null != message);
        Long userId = message.getPushReceiveScope().getUserId();
        String msgType = messageTypeProperties.convertType(message.getPayloadInfo().getType());

        addMessageTypeLastMsg(message, userId, msgType);
    }

    /**
     * 添加互动消息缓存
     *
     * @param message 推送消息
     * @param userId  目标用户
     * @param msgType 推送消息类型
     */
    private void addInteractionCache(PushMessage message, Long userId, String msgType) {
        int index = IntervalPeriodUtils.getIntervalIndex();
        //记录每个用户在时间段内最后一条互动消息
        KeyGenerator key = LAST_INTERACTION_MSG.copy().appendKey(index).appendKey(msgType);
        redisHashMapAdapter.put(key, userId.toString(), message);
        redisHashMapAdapter.expire(key, 60 * 60L);

        //记录每个用户在时间段内的互动消息总数
        key = INTERACTION_NUM.copy().appendKey(index).appendKey(msgType);
        redisHashMapAdapter.increment(key, userId.toString(), 1);
        redisHashMapAdapter.expire(key, 60 * 60L);
    }

    /**
     * 移除互动消息缓存
     *
     * @param userId  目标用户
     * @param msgType 推送消息类型
     */
    private void removeInteractionCache(Long userId, String msgType) {
        int index = IntervalPeriodUtils.getIntervalIndex();
        KeyGenerator key = LAST_INTERACTION_MSG.copy().appendKey(index).appendKey(msgType);
        redisHashMapAdapter.remove(key, userId.toString());

        key = INTERACTION_NUM.copy().appendKey(index).appendKey(msgType);
        redisHashMapAdapter.put(key, userId.toString(), 0);
    }

    private MessageType build(UserMessageBean message, String msgType) {
        MessageType messageType = new MessageType();

        messageType.setMessageType(msgType);
        messageType.setSourceLastTime(message.getAddTime());
        if (messageTypeProperties.isInteraction(msgType)) {
            JSONObject extend = parseObject(message.getLinkParam());
            String nickName = Objects.toString(extend.get(NICKNAME), "");
            String action = Objects.toString(extend.get(ACTION), "");

            messageType.setDescription(nickName + action);
        } else {
            messageType.setDescription(message.getContent());
        }

        return messageType;
    }

    /**
     * 将用户的最后一次未读消息写入到缓存。用于显示在消息列表中
     *
     * @param message 推送消息
     * @param userId  用户ID
     * @param msgType 消息中心栏目类型
     */
    private void addMessageTypeLastMsg(PushMessage message, Long userId, String msgType) {
        MessageType entity = new MessageType();
        KeyGenerator key = getLastMsgCacheKey(userId);

        //数据迁移原因，初次加载时，尝试加载历史数据
        if (Boolean.FALSE.equals(redisHashMapAdapter.hasKey(key))) {
            Map<String, MessageType> historyMap = loadLastMsgHistory(userId);
            redisHashMapAdapter.putAll(key, historyMap);
            redisHashMapAdapter.expire(key, 7 * 24 * 3600L);
        }

        if (messageTypeProperties.isInteraction(msgType)) {
            Map<String, Object> extend = message.getPayloadInfo().getExtend();
            String nickName = Objects.toString(extend.get(NICKNAME), "");
            String action = Objects.toString(extend.get(ACTION), "");
            entity.setDescription(nickName + action);
        } else {
            entity.setDescription(message.getContent());
        }

        //如果是即时通讯消息
        if (PushMessageEnum.IM_MESSAGE.name().equals(msgType)) {
            Map<String, Object> extend = message.getPayloadInfo().getExtend();

            String nickname = Objects.toString(extend.get(NICKNAME), "");
            String headImg = Objects.toString(extend.get(HEAD_IMG), "");
            String customType = Objects.toString(extend.get(CUSTOM_TYPE), "");

            entity.setImg(headImg);
            entity.setTitle(nickname);
            entity.setMessageType(customType);

            msgType = customType;
        }

        // 如果是评论或者点赞则保存点赞用户的头像
        if (COMMENT.name().equals(msgType) || LIKE.name().equals(msgType)) {
            Map<String, Object> extend = message.getPayloadInfo().getExtend();
            String nickname = Objects.toString(extend.get(ICON), "");
            //设置最新的消息的头像
            entity.setHeadIcon(nickname);
        }

        entity.setSourceLastTime(new Date());
        entity.setMessageType(msgType);
        redisHashMapAdapter.put(key, msgType, entity);
    }

    @Override
    public void updateMessageByType(Long userId, String messageType) {
        //如果是用户的消息，则不需要更新数据库
        if (!NumberUtils.isDigits(messageType)
                && !MessageTypeEnum.ADD_FUNS.name().equals(messageType)) {
            MessageListParam messageListParam = new MessageListParam();
            messageListParam.setUserId(userId);
            messageListParam.setMessageType(messageType);
            constructionParam(messageListParam);

            this.messageMapper.updateMessageByType(messageListParam);
        }

        //对应的消息类型未读数量设置为0
        redisHashMapAdapter.put(getUnreadCacheKey(userId), messageType, 0L);

        //如果互动消息已读，则定时推送互动消息移除对应的类型信息
        if (messageTypeProperties.isInteraction(messageType)) {
            removeInteractionCache(userId, messageType);
        }
        // 如果消息是点赞或者评论， 添加互动消息数量变化事件
        if (COMMENT.name().equalsIgnoreCase(messageType)
                || LIKE.name().equalsIgnoreCase(messageType)
        ) {
            //添加用户互动消息未读数量变化事件
            InteractMessageDTO interactMessageDTO = getUnReadInteractMessage(userId);

            addInteractUserEvent(userId, interactMessageDTO);
        }
        //通知客户端设置未读消息数量（补偿策略，不清楚客户端有没有对已读消息做处理）
        addUnReadUserEvent(userId, messageType);
    }

    @Override
    public void addMessage(PushMessage message, Long userId) {
        UserMessageBean entity = new UserMessageBean();
        entity.setId(sequenceCreater.nextLongId());
        entity.setTitle(removeNonBmpUnicodes(message.getTitle()));
        entity.setContent(removeNonBmpUnicodes(message.getContent()));
        entity.setUserId(userId);
        entity.setMsgType(String.valueOf(message.getPayloadInfo().getType()));
        entity.setStatus((byte) 0);
        entity.setLinkParam(removeNonBmpUnicodes(JSON.toJSONString(message.getPayloadInfo())));
        entity.setAddTime(new Date());

        addUnreadMsg(message, userId);

        if (message.isPersistence()) {
            this.messageMapper.insertSelective(entity);
        }
    }

    private String removeNonBmpUnicodes(String source) {
        if (messageTypeProperties.isFilterEmoji()) {
            return null == source ? null : source.replaceAll("[^\\u0000-\\uFFFF]", "");
        }
        return source;
    }

    @Override
    public Map<String, Long> getUnReadNum(Long userId, String messageType) {
        Map<String, Long> resultMap = Maps.newHashMap();

        //获取用户的未读消息缓存
        Map<String, Long> unReadMap = getUnreadMsgNumMap(userId);

        if (isBlank(messageType)) {
            return resultMap;
        }
        if (!unReadMap.containsKey(messageType)) {
            return resultMap;
        }
        resultMap.put(messageType, unReadMap.get(messageType));

        return getUnreadMsgNumMap(userId);
    }

    @Override
    public MessageTypeNewDTO getMessageCenterList(String areaCode, Long userId, MsgCenterNewParam basicParam) {
        MessageTypeNewDTO msgDto = new MessageTypeNewDTO();

        List<String> skipTypes = Lists.newArrayList();

        //提审时不显示余额 备注：平台：5 移除余额变动通知和系统通知 -------
        if (basicParam.getPlatform() == PlatformEnum.APPLET.getCode()
                || Boolean.TRUE.equals(appVersionIntegrationService.getPublishState(basicParam))) {
            skipTypes.add(MessageTypeEnum.BALANCE.name());
        }

        //平台：1和2 移除邀请通知
        if (basicParam.getPlatform() == PlatformEnum.ANDROID.getCode()
                || basicParam.getPlatform() == PlatformEnum.IOS.getCode()) {
            skipTypes.add(MessageTypeEnum.INVITE.name());
        }

        //设置用户的最后未读消息
        setNewMessageList(msgDto, userId, skipTypes, basicParam);

        //附加广告信息
        msgDto.setAdvertDTOS(advertIntegrationService.getAdvertByType("10", areaCode, userId));

        return msgDto;
    }

    @Override
    public Boolean hasLatestMessage(Long userId, String areaCode, Byte type, BasicParam basicParam) {
        //1、根据类型（点赞或评论）和用户id 和 areaCode获取是否,这里的key 是messageType
        Map<String, Long> unreadNumMap = getUnreadMsgNumMap(userId);
        if (type == null || userId == null) {
            return false;
        }

        // 如果是0则为评论，1则为点赞
        Long result = null;
        switch (type) {
            case 0:
                result = unreadNumMap.get(COMMENT.name());
                break;
            case 1:
                result = unreadNumMap.get(LIKE.name());
                break;
            default:
                return false;
        }
        //2、判断如果数量大于0则返回true， 如果小于0则返回false
        if (result == null) {
            return false;
        }
        return result > 0;
    }

    @Override
    public InteractMessageDTO getUnReadInteractMessage(Long userId) {

        InteractMessageDTO interactMessageDTO = new InteractMessageDTO();

        String jumpUrlFormatStr = "wst://message/commentLike?selectedTab=%s";
        // 默认跳评论tab
        interactMessageDTO.setJumpUrl(String.format(jumpUrlFormatStr, 0));

        // 开关配置，如果关闭开关则不显示互动消息，返回null
        if (Boolean.FALSE.equals(interactMessageProperties.getOpen())) {
            return interactMessageDTO;
        }

        // 1、获取最新的未读消息（评论或点赞）
        Map<String, Long> unreadNumMap = getUnreadMsgNumMap(userId);

        int messageCount = 0;

        // 未读评论数
        long commentUnReadMsgCount = getLongValue(unreadNumMap.get(COMMENT.name()));
        // 未读点赞数
        long likeUnReadMsgCount = getLongValue(unreadNumMap.get(LIKE.name()));

        messageCount += commentUnReadMsgCount;
        messageCount += likeUnReadMsgCount;


        log.info("用户id: {}, 评论消息数量：{},点赞消息数量：{}", userId, unreadNumMap.get(COMMENT.name()), unreadNumMap.get(LIKE.name()));

        // 2、将未读消息点赞和评论数量相加设置未读互动消息数量
        interactMessageDTO.setMessageCount(messageCount);

        // 如果未读消息数量为0，则直接返回
        if (messageCount <= 0) {
            return interactMessageDTO;
        }

        // 3、获取最新的一条未读消息（可能是评论或者点赞），最后一条未读的评论或互动消息

        // 获取各种类型的最后一条消息数据
        Map<String, MessageType> lastMsgMap = getLastMsgMap(userId);

        MessageType commentLastMsg = lastMsgMap.get(COMMENT.name());
        MessageType likeLastMsg = lastMsgMap.get(LIKE.name());

        if (likeUnReadMsgCount <= 0 && commentUnReadMsgCount > 0) {
            interactMessageDTO.setLatestMessageType((byte) 0);
            if (commentLastMsg != null) {
                String headIcon = commentLastMsg.getHeadIcon();
                // 5、设置最新的消息的头像
                interactMessageDTO.setHeadImageUrl(StringUtils.isNotBlank(headIcon) ? headIcon : interactMessageProperties.getDefaultHeadImg());
            }
            return interactMessageDTO;
        }

        if (commentUnReadMsgCount <= 0 && likeUnReadMsgCount > 0) {
            interactMessageDTO.setLatestMessageType((byte) 1);
            if (likeLastMsg != null) {
                String headIcon = likeLastMsg.getHeadIcon();
                // 5、设置最新的消息的头像
                interactMessageDTO.setJumpUrl(String.format(jumpUrlFormatStr, 1));
                interactMessageDTO.setHeadImageUrl(StringUtils.isNotBlank(headIcon) ? headIcon : interactMessageProperties.getDefaultHeadImg());
            }
            return interactMessageDTO;
        }

        if (likeUnReadMsgCount > 0 && commentUnReadMsgCount > 0) {

            if (commentLastMsg != null && likeLastMsg != null) {
                Date commentCreateTime = commentLastMsg.getSourceLastTime();
                Date likeCreateTime = likeLastMsg.getSourceLastTime();

                //如果评论时间在点赞之后
                if (DateUtils.after(commentCreateTime, likeCreateTime)) {
                    interactMessageDTO.setLatestMessageType((byte) 0);
                    String headIcon = commentLastMsg.getHeadIcon();
                    // 5、设置最新的消息的头像
                    interactMessageDTO.setHeadImageUrl(StringUtils.isNotBlank(headIcon) ? headIcon : interactMessageProperties.getDefaultHeadImg());
                } else {
                    interactMessageDTO.setLatestMessageType((byte) 1);
                    //如果最新消息是的点赞则跳点赞tab，评论则跳评论tab，指定跳转协议
                    interactMessageDTO.setJumpUrl(String.format(jumpUrlFormatStr, 1));

                    String headIcon = likeLastMsg.getHeadIcon();
                    // 5、设置最新的消息的头像
                    interactMessageDTO.setHeadImageUrl(StringUtils.isNotBlank(headIcon) ? headIcon : interactMessageProperties.getDefaultHeadImg());
                }
            }
            return interactMessageDTO;
        }
        log.info("用户互动消息信息：用户id-{}, 互动消息信息：{}", userId, JSON.toJSONString(interactMessageDTO));
        return interactMessageDTO;
    }

    /**
     * 获取Long的值
     *
     * @param value value
     * @return value
     */
    public long getLongValue(Long value) {
        return value == null ? 0 : value;
    }

    /**
     * 获取消息中心显示的类型、标题、图标、未读数量。用于显示消息中心的列表
     *
     * @param msgDto    消息传输对象
     * @param userId    用户ID
     * @param skipTypes 不显示的消息类型
     */
    private void setNewMessageList(MessageTypeNewDTO msgDto, Long userId, List<String> skipTypes, BasicParam basicParam) {
        Map<String, Long> unreadNumMap = getUnreadMsgNumMap(userId);
        Map<String, MessageType> lastMsgMap = getLastMsgMap(userId);

        List<MessageType> msgList = lastMsgMap.values()
                .stream()
                .sorted((pre, next) ->
                        dateTimeComparator.compare(next.getSourceLastTime(), pre.getSourceLastTime()))
                .collect(Collectors.toList());

        List<MessageType> afterFilter = Lists.newArrayList();

        for (MessageType msgType : msgList) {
            String type = msgType.getMessageType();
            //跳过不在特定平台显示的消息
            if (null == type || skipTypes.contains(msgType.getMessageType())) {
                continue;
            }

            //三种特殊的消息类型处理,这里排除的类型，包含点赞、评论、新增粉丝、今日热闻，点赞和评论合并到了一起
            if (StringUtils.equalsAny(type, LIKE.name(), COMMENT.name(), ADD_FUNS.name(), NEWS.name())) {
                continue;
            }

            // 数字类型的为私聊消息，不进行处理
            if (NumberUtils.isDigits(type)) {
                continue;
            }

            //处理日期格式
            if (null != msgType.getSourceLastTime()) {
                msgType.setLastTime(DateFormatUtils.format(msgType.getSourceLastTime()));
            }

            msgType.setUnreadNum(getUnreadIntVal(unreadNumMap, type));

            MessageType configTypeInfo = messageTypeProperties.getMessageInfo(type);

            if (log.isDebugEnabled()) {
                log.debug("match msg type:{}", configTypeInfo);
            }

            if (null != configTypeInfo) {
                msgType.setTitle(configTypeInfo.getTitle());
                msgType.setImg(configTypeInfo.getImg());
            } else {
                //未匹配的类型不再显示
                continue;
            }

            afterFilter.add(msgType);
        }
        msgDto.setMessageTypes(afterFilter);
        newFillNull(msgDto, unreadNumMap);
    }

    /**
     * 点赞、评论、新增粉丝在某一个版本调整出了消息栏目到单独的顶部区域
     *
     * @param msgDto 消息栏目信息完整信息
     */
    private void newFillNull(MessageTypeNewDTO msgDto, Map<String, Long> unreadNumMap) {
        MessageType commentAndLikeType = new MessageType();
        commentAndLikeType.setMessageType(LIKE_AND_COMMENT.name());
        commentAndLikeType.setTitle("评论点赞");
        // 将点赞和评论数量相加
        commentAndLikeType.setUnreadNum(getUnreadIntVal(unreadNumMap, LIKE.name()) + getUnreadIntVal(unreadNumMap, COMMENT.name()));
        msgDto.setCommentAndLikeType(commentAndLikeType);

        MessageType followType = new MessageType();
        followType.setMessageType(ADD_FUNS.name());
        followType.setTitle("新增粉丝");
        followType.setUnreadNum(getUnreadIntVal(unreadNumMap, ADD_FUNS.name()));
        msgDto.setFollowType(followType);

        MessageType newsType = new MessageType();
        newsType.setMessageType(NEWS.name());
        newsType.setTitle("今日热闻");
        newsType.setUnreadNum(getUnreadIntVal(unreadNumMap, NEWS.name()));
        msgDto.setTodayNews(newsType);

        if (msgDto.getMessageTypes() == null) {
            msgDto.setMessageTypes(Lists.newArrayList());
        }

        // 进行空值的填充，对消息列表进行填充
        for (String type : messageTypeProperties.getMessageTypes()) {
            // 新增消息列表中不能含有今日热闻，今日热闻被提到顶部三个图标模块中
            if (StringUtils.equalsAny(type, LIKE.name(), COMMENT.name(), ADD_FUNS.name(), NEWS.name())) {
                continue;
            }
            if (msgDto.getMessageTypes().stream().anyMatch(item -> type.equals(item.getMessageType()))) {
                continue;
            }
            MessageType innerType = new MessageType();
            MessageType configTypeInfo = messageTypeProperties.getMessageInfo(type);
            if (null != configTypeInfo && null != configTypeInfo.getTitle()) {
                innerType.setMessageType(type);
                innerType.setTitle(configTypeInfo.getTitle());
                innerType.setImg(configTypeInfo.getImg());

                msgDto.getMessageTypes().add(innerType);
            }
        }
    }

    /**
     * 添加用户事件，用于显示未读消息的小红点
     *
     * @param userId  用户ID
     * @param msgType 消息类型
     */
    private void addUnReadUserEvent(Long userId, String msgType) {
        /*
         * V3.4.0 不对3.0.0版本和小程序进行兼容，减少无谓的用户事件
         */
        if (MessageTypeEnum.BALANCE.name().equals(msgType)) {
            //余额变动仅在客户端显示
            //userEventService.add(buildPushPayloadInfo(userId));
        } else if (MessageTypeEnum.INVITE.name().equals(msgType)) {
            //邀请信息仅在小程序显示
            //userEventService.add(buildAppletPushPayloadInfo(userId));
        } else if (PushMessageEnum.IM_MESSAGE.name().equals(msgType)) {
            //用户私聊消息
            //userEventService.add(buildPushPayloadInfo(userId));
        } else {
            //系统消息
            userEventService.add(buildSystemPush(userId));
            //userEventService.add(buildPushPayloadInfo(userId));
            //userEventService.add(buildAppletPushPayloadInfo(userId));
        }
    }

    /**
     * 构建未读消息总数
     *
     * @param userId 用户ID
     * @return 获取用户的未读消息总数，推送给用户事件
     */
    private PushPayloadInfo buildSystemPush(Long userId) {
        return PushPayloadInfo.build().setType(UserEventEnum.UN_READ_SYSTEM_MSG.type)
                .addExtend("userId", userId).addExtend("current", getUnReadMsg(userId, 0, "3.0.0"));
    }

    /**
     * app用户事件
     *
     * @param userId 用户ID
     * @deprecated 不再对3.0.0版本之前进行兼容
     */
    @Deprecated
    private PushPayloadInfo buildPushPayloadInfo(Long userId) {
        return PushPayloadInfo.build().setType(UserEventEnum.USER_UN_READ_MSG.type)
                .addExtend("userId", userId).addExtend("current", getUnReadMsg(userId, 0, null));
    }

    /**
     * 小程序用户事件
     *
     * @param userId 用户ID
     * @deprecated 小程序未推广，不再推送消息
     */
    @Deprecated
    private PushPayloadInfo buildAppletPushPayloadInfo(Long userId) {
        return PushPayloadInfo.build().setType(UserEventEnum.APPLET_USER_UN_READ_MSG.type)
                .addExtend("userId", userId).addExtend("current", getUnReadMsg(userId, 5, null));
    }

    /**
     * 推送小程序模板消息
     *
     * @param pushMessage 推送消息
     * @param userId      目标用户ID
     */
    private void pushTemplateMessage(PushMessage pushMessage, Long userId) {
        int type = pushMessage.getPayloadInfo().getType();
        //判断消息类型是否需要推送小程序模板消息
        if (messageTypeProperties.getTemplate().contains(type)) {
            PushMessage cloneMsg = PushMessage.build();
            BeanUtils.copyProperties(pushMessage, cloneMsg);

            cloneMsg.setPushReceiveScope(PushReceiveScope.pushRule(PushReceiverRuleEnum.DEFAULT)
                    .addRuleParam(ReceiverRuleParamConstant.USER_IDS, userId));
            pushTemplateMessageIntegrationService.pushTemplateMessage(cloneMsg);
        }
    }

    /**
     * 构造参数信息
     * 根据查询的消息聚合类型获取对应的子类型
     */
    private void constructionParam(MessageListParam param) {
        //备注：小程序版本1.0.0和万事通2.3.0版本互动消息变成评论通知和点赞通知
        if (isNotEmpty(param.getMessageType())
                && MessageTypeEnum.INTERACTION.name().equals(param.getMessageType())) {
            List<String> msgTypeList = messageTypeProperties.getSubTypes(MessageTypeEnum.COMMENT.name());
            msgTypeList.addAll(messageTypeProperties.getSubTypes(LIKE.name()));
            param.setMsgTypeList(msgTypeList);
        } else {
            param.setMsgTypeList(messageTypeProperties.getSubTypes(param.getMessageType()));
        }
    }
}
