package com.bxm.localnews.news.controller;

import com.bxm.localnews.common.annotation.TouTiaoAuth;
import com.bxm.localnews.common.constant.ErrorCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.news.domain.NewsKindMapper;
import com.bxm.localnews.news.domain.UserKindMapper;
import com.bxm.localnews.news.param.UserKindExample;
import com.bxm.localnews.news.service.NewsKindService;
import com.bxm.localnews.news.vo.KindVo;
import com.bxm.localnews.news.vo.NewsKind;
import com.bxm.localnews.news.vo.UserKind;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

@Api(tags = "4-05 新闻频道接口", description = "用户对频道的操作")
@RequestMapping("api/kind")
@RestController
public class KindController {

    @Resource
    private NewsKindMapper newsKindMapper;

    @Resource
    private UserKindMapper userKindMapper;

    @Resource
    private NewsKindService newsKindService;

    @ApiOperation(value = "4-05-01 我的频道", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id", required = true)
    })
    @TouTiaoAuth
    @GetMapping("myKinds")
    public Json<KindVo> myKinds(Long userId) {
        //XXX [zhaoyadong]时间：直接mapper层查询
        List<NewsKind> allKinds = this.newsKindMapper.selectAllKinds();
        List<NewsKind> myKinds = this.userKindMapper.myKinds(userId);
        KindVo kindVo = new KindVo(allKinds, myKinds);
        return ResultUtil.genSuccessResult(kindVo);
    }

    @ApiOperation(value = "4-05-02 添加频道", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id", required = true),
            @ApiImplicitParam(name = "kindId", value = "频道id", required = true)
    })
    @TouTiaoAuth
    @PostMapping("add")
    public Json add(Long userId, Integer kindId) {
        if (kindId == null) {
            return ResultUtil.genFailedResult(ErrorCode.PARAM_ERR, "参数错误");
        }
        try {
            this.userKindMapper.add(userId, kindId);
        } catch (DuplicateKeyException e) {
            return ResultUtil.genFailedResult(ErrorCode.DUPLICATE, "重复添加");
        }
        return ResultUtil.genSuccessResult();
    }

    @ApiOperation(value = "4-05-03 频道排序", notes = "对用户添加的频道进行排序", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id", required = true)
    })
    @TouTiaoAuth
    @PostMapping("resort")
    public Json resort(Long userId, Integer[] kindIds) {
        try {
            this.userKindMapper.deleteByExample(new UserKindExample() {{
                this.createCriteria().andUserIdEqualTo(userId);
            }});
            if (kindIds != null && kindIds.length > 0) {
                List<UserKind> userKindList = new ArrayList<>();
                for (int i = 0; i < kindIds.length; i++) {
                    UserKind userKind = new UserKind();
                    userKind.setUserId(userId);
                    userKind.setKindId(kindIds[i]);
                    userKind.setSortNo(i);
                    userKindList.add(userKind);
                }
                this.userKindMapper.batchInsert(userKindList);
            }
        } catch (DuplicateKeyException e) {
            ResultUtil.genFailedResult(ErrorCode.DUPLICATE, "重复添加");
        }
        return ResultUtil.genSuccessResult();
    }

    @ApiOperation(value = "4-05-04 所有频道", httpMethod = "GET")
    @GetMapping("newsKind")
    public Json<List<NewsKind>> newsKind() {
        List<NewsKind> allKinds = this.newsKindService.selectAll();
        return ResultUtil.genSuccessResult(allKinds);
    }

}
