package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.news.config.UserProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.dto.BaseForumPostDTO;
import com.bxm.localnews.news.dto.ForumPostBriefInfoDto;
import com.bxm.localnews.news.dto.ImgDTO;
import com.bxm.localnews.news.enums.PostTypeEnum;
import com.bxm.localnews.news.factory.impl.ExtendFactory;
import com.bxm.localnews.news.service.ForumPostFacadeService;
import com.bxm.localnews.news.service.ForumPostService;
import com.bxm.localnews.news.service.ForumPostStatisticService;
import com.bxm.localnews.news.util.OssTransCodeUtils;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

@Service
public class ForumPostFacadeServiceImpl implements ForumPostFacadeService {

    private final ForumPostService forumPostService;

    private final ForumPostStatisticService forumPostStatisticService;

    private final ForumPostMapper forumPostMapper;

    private final RedisSetAdapter redisSetAdapter;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserProperties userProperties;

    private final OssTransCodeUtils ossTransCodeUtils;

    @Autowired
    public ForumPostFacadeServiceImpl(ForumPostService forumPostService,
                                      ForumPostStatisticService forumPostStatisticService,
                                      ForumPostMapper forumPostMapper,
                                      RedisSetAdapter redisSetAdapter,
                                      RedisHashMapAdapter redisHashMapAdapter,
                                      UserProperties userProperties,
                                      OssTransCodeUtils ossTransCodeUtils) {
        this.forumPostService = forumPostService;
        this.forumPostStatisticService = forumPostStatisticService;
        this.forumPostMapper = forumPostMapper;
        this.redisSetAdapter = redisSetAdapter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.userProperties = userProperties;
        this.ossTransCodeUtils = ossTransCodeUtils;
    }

    @Override
    public ForumPostBriefInfoDto getBriefInfo(Long postId, Long userId, String areaCode, String ip) {
        ForumPostVo forumPostVo = forumPostService.getForumPostDetail(postId, userId, null, new BasicParam(), areaCode, ip);
        ForumPostBriefInfoDto briefInfoDto = new ForumPostBriefInfoDto();

        if (null != forumPostVo) {
            String title = forumPostVo.getTitle();
            if (StringUtils.isBlank(title) || PostTypeEnum.NOTE.getCode() == forumPostVo.getPostType()) {
                title = StringUtils.substring(forumPostVo.getTextField(), 0, 50);
            }
            PostImgVo imgInfo = forumPostVo.getShareImg();
            if (null != imgInfo) {
                briefInfoDto.setConvertImgUrl(imgInfo.getImgUrl());
            }
            briefInfoDto.setHasVote(forumPostVo.getVoteDetailDTO() != null);
            briefInfoDto.setId(postId);
            briefInfoDto.setTitle(title);
        }

        return briefInfoDto;
    }

    @Override
    public Integer getPublishPostNum(Long userId) {
        return forumPostMapper.selectPublishPostNumByUserId(userId);
    }

    @Override
    public List<ForumPostClickCountVo> getRecentPosts(Integer clickCountLimit) {
        return forumPostMapper.getRecentPosts(clickCountLimit, DateUtils.addField(new Date(), Calendar.DAY_OF_MONTH, -1));
    }

    @Override
    public void batchAddClick(List<ForumPostClickCountVo> forumPostClickList) {
        if (CollectionUtils.isNotEmpty(forumPostClickList)) {
            MybatisBatchBuilder.create(ForumPostMapper.class, forumPostClickList).run(ForumPostMapper::batchAddClick);
        }
    }

    @Async
    @Override
    public void transcodePostVideo(Long id, List<PostImgVo> postImgList) {
        if (CollectionUtils.isEmpty(postImgList)) {
            ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(id);
            if (forumPostVo != null) {
                postImgList = forumPostVo.getPostImgList();
            }
        }

        if (CollectionUtils.isNotEmpty(postImgList)) {
            postImgList.forEach(e -> {
                if ("VIDEO".equals(e.getType())) {
                    ossTransCodeUtils.transCodeByOriginUrl(e.getVideoUrl(), id);
                }
            });
        }
    }

    @Override
    public void batchUpdatePostVideo(List<TranscodeMap> transcodeMapList) {
        if (CollectionUtils.isNotEmpty(transcodeMapList)) {
            MybatisBatchBuilder.create(ForumPostMapper.class, transcodeMapList).run(ForumPostMapper::batchUpdatePostVideo);
        }
    }

    @Override
    public BaseForumPostDTO getForumPostByUserId(Long userId, Long targetUserId) {
        KeyGenerator recentKey = RedisConfig.RECENT_DEVELOPMENTS.copy().appendKey(String.valueOf(userId));
        //每次取目标用户的第一条帖子
        ForumPostVo forumPostVo = forumPostMapper.selectPostByUserIdAndWeek(targetUserId);
        BaseForumPostDTO result = null;
        //若用户的最新的第一条帖子已被获取,则返回空
        boolean flag = Objects.nonNull(forumPostVo) && !redisSetAdapter.exists(recentKey, forumPostVo.getId());
        if (flag) {
            List<PostImgVo> coverList = forumPostStatisticService.getCoverFromPost(
                    forumPostVo.getCoverList(), forumPostVo.getPostImgList(), forumPostVo.getCoverSelect());
            result = BaseForumPostDTO.builder()
                    .id(forumPostVo.getId())
                    //分享图片为空时取默认分享图
                    .imgUrl(CollectionUtils.isEmpty(coverList) ? null : coverList.get(0).getImgUrl())
                    .textField(forumPostVo.getTextField())
                    .title(ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField()))
                    .publishTime(forumPostVo.getPublishTime())
                    .build();
            redisSetAdapter.add(recentKey, forumPostVo.getId());
            //缓存7天后过期
            redisSetAdapter.expire(recentKey, DateUtils.WEEK_MILLISECOND / 1000);
        }
        return result;
    }

    @Override
    public List<UserImgVo> batchGetUserImg(List<Long> userIdList, Long currentUserId) {
        List<UserImgVo> userImgVoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return userImgVoList;
        }

        boolean hasCurrent = false;
        if (currentUserId != null && userIdList.contains(currentUserId)) {
            userIdList.remove(currentUserId);
            hasCurrent = true;
        }

        if (CollectionUtils.isNotEmpty(userIdList)) {
            //先取缓存
            Set<String> userIds = userIdList.stream().map(Object::toString).collect(Collectors.toSet());
            List<String> userForumImgCache = redisHashMapAdapter.multiGet(RedisConfig.USER_FORUM_IMG, userIds, String.class);

            //处理缓存数据
            if (CollectionUtils.isNotEmpty(userForumImgCache)) {
                userForumImgCache.forEach(userCache -> {
                    UserImgVo userImgVo = JSON.parseObject(userCache, UserImgVo.class);
                    if (null != userImgVo) {
                        userIdList.remove(userImgVo.getUserId());
                        userImgVoList.add(userImgVo);
                    }
                });
            }

            //若缓存中无法取到所有用户图片信息
            if (CollectionUtils.isNotEmpty(userIdList)) {
                userImgVoList.addAll(batchLoadUserImgVoToRedis(userIdList));
            }
        }

        //当前用户需返回仅自己可见内容
        if (hasCurrent) {
            List<ForumPostVo> forumPostVoList = forumPostMapper.batchGetUserPosts(
                    Collections.singletonList(currentUserId), userProperties.getAdvertTopicId(), 1);
            UserImgVo userImgVo = convertPostToUserImg(forumPostVoList, currentUserId);
            if (userImgVo != null) {
                userImgVoList.add(userImgVo);
            }
        }

        return userImgVoList;
    }

    @Override
    public void cacheUserPostImg(Date startTime) {
        //获取有更新的帖子用户
        List<Long> userIdList = forumPostMapper.getUpdatedUser(startTime);
        //缓存数据
        batchLoadUserImgVoToRedis(userIdList);
    }

    private List<UserImgVo> batchLoadUserImgVoToRedis(List<Long> userIdList) {
        List<UserImgVo> userImgVoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return userImgVoList;
        }

        //获取用户帖子
        List<ForumPostVo> forumPostVoList = forumPostMapper.batchGetUserPosts(userIdList, userProperties.getAdvertTopicId(), 0);
        if (CollectionUtils.isEmpty(forumPostVoList)) {
            return userImgVoList;
        }

        //映射用户帖子图片
        Map<Long, List<ForumPostVo>> userForumPostMap = forumPostVoList.stream().collect(Collectors.groupingBy(ForumPostVo::getUserId));
        Map<String, String> userImgVoMap = new HashMap<>();
        userIdList.forEach(userId -> {
            UserImgVo userImgVo = convertPostToUserImg(userForumPostMap.get(userId), userId);
            //即使为空，也要覆盖缓存
            userImgVoMap.put(userId.toString(), JSON.toJSONString(userImgVo));
            if (null != userImgVo) {
                userImgVoList.add(userImgVo);
            }
        });

        //存入缓存
        if (userImgVoMap.size() > 0) {
            redisHashMapAdapter.putAll(RedisConfig.USER_FORUM_IMG, userImgVoMap);
        }

        return userImgVoList;
    }

    private UserImgVo convertPostToUserImg(List<ForumPostVo> forumPostVos, Long userId) {
        if (CollectionUtils.isEmpty(forumPostVos)) {
            return null;
        }

        // 只获取4个
        int size = 0;
        List<ImgDTO> imgDtoList = new ArrayList<>();
        for (ForumPostVo forumPostVo : forumPostVos) {
            List<PostImgVo> postImgVoList = forumPostVo.getPostImgList();
            if (CollectionUtils.isNotEmpty(postImgVoList)) {
                for (PostImgVo postImgVo : postImgVoList) {
                    if ("IMG".equals(postImgVo.getType())) {
                        ImgDTO imgDto = new ImgDTO();
                        imgDto.setId(forumPostVo.getId());
                        imgDto.setCategory("POST");
                        imgDto.setType(postImgVo.getType());
                        imgDto.setImgUrl(postImgVo.getImgUrl());
                        imgDtoList.add(imgDto);
                        size++;
                        if (size >= 4) {
                            break;
                        }
                    }
                }
            }
            if (size >= 4) {
                break;
            }
        }

        UserImgVo userImgVo = new UserImgVo();
        userImgVo.setUserId(userId);
        userImgVo.setImgDtoList(imgDtoList);
        return userImgVo;
    }

}
