package com.bxm.localnews.news.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.dto.MixRecommendDTO;
import com.bxm.localnews.integration.AppVersionIntegrationService;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.integration.NewsRecommendIntegrationService;
import com.bxm.localnews.integration.NewsSearchIntegrationService;
import com.bxm.localnews.news.config.NewsProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.convert.impl.ForumPostConvert;
import com.bxm.localnews.news.convert.impl.NewsRecommendConvert;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.MixedRecommendPoolMapper;
import com.bxm.localnews.news.dto.ForumPostDTO;
import com.bxm.localnews.news.dto.VideoDto;
import com.bxm.localnews.news.param.HomeRecommendParam;
import com.bxm.localnews.news.param.VideoQueryParam;
import com.bxm.localnews.news.service.*;
import com.bxm.localnews.news.util.VersionUtils;
import com.bxm.localnews.news.vo.*;
import com.bxm.localnews.param.MixRecommendParam;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class RecommendServiceImpl extends BaseService implements RecommendService {

    private MixedRecommendPoolMapper mixedRecommendPoolMapper;

    private NewsRecommendConvert newsRecommendConvert;

    private AppVersionIntegrationService appVersionIntegrationService;

    private NewsProperties newsProperties;

    private VideoRecommendService videoRecommendService;

    private RedisStringAdapter redisStringAdapter;

    private NewsSearchIntegrationService newsSearchIntegrationService;

    private NewsService newsService;

    private NewsRecommendIntegrationService newsRecommendIntegrationService;

    private ForumPostMapper forumPostMapper;

    private LocationIntegrationService locationIntegrationService;

    private ForumPostService forumPostService;

    private ForumTopicService forumTopicService;

    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    public RecommendServiceImpl(MixedRecommendPoolMapper mixedRecommendPoolMapper,
                                NewsRecommendConvert newsRecommendConvert,
                                AppVersionIntegrationService appVersionIntegrationService,
                                NewsProperties newsProperties,
                                VideoRecommendService videoRecommendService,
                                RedisStringAdapter redisStringAdapter,
                                NewsSearchIntegrationService newsSearchIntegrationService,
                                NewsService newsService,
                                NewsRecommendIntegrationService newsRecommendIntegrationService,
                                ForumPostMapper forumPostMapper,
                                LocationIntegrationService locationIntegrationService,
                                ForumPostService forumPostService,
                                ForumTopicService forumTopicService,
                                RedisHashMapAdapter redisHashMapAdapter) {
        this.mixedRecommendPoolMapper = mixedRecommendPoolMapper;
        this.newsRecommendConvert = newsRecommendConvert;
        this.appVersionIntegrationService = appVersionIntegrationService;
        this.newsProperties = newsProperties;
        this.videoRecommendService = videoRecommendService;
        this.redisStringAdapter = redisStringAdapter;
        this.newsSearchIntegrationService = newsSearchIntegrationService;
        this.newsService = newsService;
        this.newsRecommendIntegrationService = newsRecommendIntegrationService;
        this.forumPostMapper = forumPostMapper;
        this.locationIntegrationService = locationIntegrationService;
        this.forumPostService = forumPostService;
        this.forumTopicService = forumTopicService;
        this.redisHashMapAdapter = redisHashMapAdapter;
    }

    @Override
    public Json<NewsMeta> execRecommend(NewsRecommendParam param, BasicParam basicParam) {
        logger.debug("[oldExecRecommend]新闻推荐请求+1：{}", param);
        int count = 0;
        return retry(param, count, basicParam);

    }

    private Json<NewsMeta> retry(NewsRecommendParam param, int count, BasicParam basicParam) {
        long begin = System.currentTimeMillis();
        NewsMeta meta = new NewsMeta();
        List<Long> idList;
        idList = newsRecommendIntegrationService.recommendList(param.getUserId(),
                param.getKindId().intValue(),
                param.getPagesize(),
                param.getAreaCode(),
                param.getCurPage());

        logger.debug("[oldExecRecommend]用户id:{},推荐引擎返回:{},耗时:{}", param.getUserId(), idList, System.currentTimeMillis() - begin);

        if (CollectionUtils.isEmpty(idList)) {
            return ResultUtil.genSuccessResult(meta);
        }
        //从es中获取新闻列表信息
        List<News> list = newsSearchIntegrationService.listRecommendedNews(idList.toArray(new Long[0]));
        logger.debug("[oldExecRecommend]从es中获取新闻列表信息,耗时:{}", System.currentTimeMillis() - begin);

        //包装新闻
        newsService.fillExtInfo(param.getUserId(),list, param.getAreaCode());
        //塞进新闻返回实体
        List<News4Client> news4ClientList = list.stream().map(news -> new News4Client(news, null)).collect(Collectors.toList());

        //添加小视频
        if (CollectionUtils.isNotEmpty(news4ClientList) && !getPublishState(basicParam)) {
            if (param.getKindId() == null || this.newsProperties.getRecommendKindId() == param.getKindId()) {
                addRecommendVideo(param.getUserId(), param.getCurPage(), news4ClientList);
            }
        }
        logger.debug("[oldExecRecommend]小视频添加,耗时:{}", System.currentTimeMillis() - begin);

        meta.setList(news4ClientList);
        meta.setNewsCount(meta.getList().size());
        // 隐藏敏感信息
        hideInfo(param, meta);
        logger.debug("[oldExecRecommend]拉取新闻列表完毕,参数:{},耗时：{}", JSONObject.toJSONString(meta), System.currentTimeMillis() - begin);

        return ResultUtil.genSuccessResult(meta);
    }

    /**
     * 推荐小视频，每2页显示一次小视频
     *
     * @param userId          用户id
     * @param curPage         当前页数
     * @param news4ClientList
     */
    private void addRecommendVideo(Long userId, Integer curPage, List<News4Client> news4ClientList) {
        logger.info("[addRecommendVideo]首页添加小视频，参数->userId:{},curPage:{}", userId, curPage);
        if (curPage != 0 && curPage % 2 == 0) {
            VideoQueryParam videoParam = new VideoQueryParam();
            videoParam.setUserId(userId);
            // 在新闻列表中间插入
            List<VideoDto> videoDtoList = videoRecommendService.execRecommend(videoParam);
            if (CollectionUtils.isNotEmpty(videoDtoList)) {
                int size = news4ClientList.size();
                news4ClientList.add(size / 2, new News4Client(videoDtoList));
            }
        }
    }

    /**
     * 处于提包状态时，隐藏部分敏感信息，用于上包 去除作者信息
     *
     * @param param 推荐参数
     * @param meta  返回结果
     */
    private void hideInfo(BasicParam param, NewsMeta meta) {
        if (appVersionIntegrationService.getPublishState(param)) {
            meta.getList().removeIf(client -> client.getVideos() != null);
            for (News4Client client : meta.getList()) {
                if (client.getNews() != null) {
                    client.getNews().setAuthor(null);
                }
            }
        }
    }

    /**
     * 查询置顶新闻
     */
    @Override
    public Json<NewsMeta> getTopNewsList(NewsTopListParam param) {
        // 置顶新闻列表
        List<News> topNewsList;

        // 区域性性置顶新闻
        List<News> topNewsOfAreaList = new ArrayList<>();
        KeyGenerator keyGeneratorOfArea = RedisConfig.TOP_NEW_OF_AREA_LIST.copy().appendKey(param.getAreaCode());
        String topNewsOfAreaListStr = redisStringAdapter.getString(keyGeneratorOfArea);
        if (StringUtils.isNotBlank(topNewsOfAreaListStr)) {
            topNewsOfAreaList = JSON.parseArray(topNewsOfAreaListStr, News.class);
        } else {
            if (param.getAreaCode() != null) {
                topNewsOfAreaList = newsSearchIntegrationService.listTopNews(5, param.getAreaCode());
                if (topNewsOfAreaList != null && topNewsOfAreaList.size() > 0) {
                    redisStringAdapter.set(keyGeneratorOfArea, JSON.toJSONString(topNewsOfAreaList));
                }
            } else {
                redisStringAdapter.set(keyGeneratorOfArea, "[]");
                redisStringAdapter.expire(keyGeneratorOfArea, 10);
            }

        }

        // 全国性置顶新闻
        List<News> topNewsOfWholeCountryList;
        KeyGenerator keyGeneratorOfWholeCountry = RedisConfig.TOP_NEW_OF_WHOLE_COUNTRY_LIST.copy();
        String topNewsOfWholeCountryListStr = redisStringAdapter.getString(keyGeneratorOfWholeCountry);
        if (StringUtils.isNotBlank(topNewsOfWholeCountryListStr)) {
            topNewsOfWholeCountryList = JSON.parseArray(topNewsOfWholeCountryListStr, News.class);
        } else {
            topNewsOfWholeCountryList = newsSearchIntegrationService.listTopNews(5, null);
            if (topNewsOfWholeCountryList != null && topNewsOfWholeCountryList.size() > 0) {
                redisStringAdapter.set(keyGeneratorOfWholeCountry, JSON.toJSONString(topNewsOfWholeCountryList));
            } else {
                redisStringAdapter.set(keyGeneratorOfWholeCountry, "[]");
                redisStringAdapter.expire(keyGeneratorOfWholeCountry, 10);
            }
        }

        NewsMeta newsMeta = new NewsMeta();
        if (CollectionUtils.isEmpty(topNewsOfAreaList) && CollectionUtils.isEmpty(topNewsOfWholeCountryList)) {
            logger.debug("当前未配置置顶新闻");
            return ResultUtil.genSuccessResult(newsMeta);
        }

        topNewsList = this.filterTopNews(topNewsOfAreaList, topNewsOfWholeCountryList, param);

        List<News> newTopNewsList = newsSearchIntegrationService.listRecommendedNews(topNewsList.stream()
                .map(News::getId)
                .toArray(Long[]::new));


        if (CollectionUtils.isEmpty(newTopNewsList)) {
            return ResultUtil.genSuccessResult(newsMeta);
        }

        // 组装结构数据
        newsService.fillExtInfo(param.getUserId(), newTopNewsList, param.getAreaCode());

        List<News4Client> news4ClientList = newTopNewsList.stream().map(news -> new News4Client(news, null)).collect(Collectors.toList());

        newsMeta.setList(news4ClientList);
        newsMeta.setNewsCount(newsMeta.getList().size());
        // 隐藏敏感信息
        this.hideInfo(param, newsMeta);

        return ResultUtil.genSuccessResult(newsMeta);
    }


    /**
     * 填充置顶新闻
     * 置顶条数限制改为最多显示两条
     * 当本地置顶和全国置顶都存在时显示一条本地一条全国，最新编辑的优先展示
     * 当只有本地资讯时展示最多两条本地置顶，最新编辑的优先展示
     * 当只有全国资讯时展示最多两条全国置顶，最新编辑的优先展示
     *
     * @param topNewsOfAreaList         区域置顶新闻
     * @param topNewsOfWholeCountryList 全国置顶新闻
     * @return 合并处理过的置顶新闻
     */
    private List<News> filterTopNews(List<News> topNewsOfAreaList, List<News> topNewsOfWholeCountryList, NewsTopListParam param) {
        List<News> topNewsList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(topNewsOfAreaList) && CollectionUtils.isNotEmpty(topNewsOfWholeCountryList)) {
            topNewsList.add(topNewsOfAreaList.get(0));
            topNewsList.add(topNewsOfWholeCountryList.get(0));
            return topNewsList;
        } else if (CollectionUtils.isNotEmpty(topNewsOfAreaList) && CollectionUtils.isEmpty(topNewsOfWholeCountryList)) {
            topNewsList.addAll(topNewsOfAreaList);
        } else if (CollectionUtils.isNotEmpty(topNewsOfWholeCountryList) && CollectionUtils.isEmpty(topNewsOfAreaList)) {
            topNewsList.addAll(topNewsOfWholeCountryList);
        }
        topNewsList = topNewsList.stream().limit(2).collect(Collectors.toList());


        return topNewsList;
    }

    @Override
    public Json<NewsMeta> listMixRecommend(HomeRecommendParam param, BasicParam basicParam) {
        Json<NewsMeta> result;

        boolean isPublishState = getPublishState(basicParam);
        //如果是不在提审且开通社区
        if (!isPublishState && checkIsNewVersionRecommend(param) ) {
            result = listMixRecommendByPromotion(param, basicParam);
        } else {
            NewsRecommendParam newsRecommendParam = newsRecommendConvert.convert(param);
            result = execRecommend(newsRecommendParam, basicParam);
        }

        if (result != null && result.getResult() != null) {
            List<News4Client> list = result.getResult().getList();
            //若数量大于5且版本号大于2.5.0，则插入推荐话题
            if (list != null && list.size() >= 5 && VersionUtils.isHighVersion(basicParam.getCurVer(), "2.5.0") > 0) {
                //插入推荐话题
                List<TopicVo> topicVoList = forumTopicService.getTopicList(param.getAreaCode());
                if (CollectionUtils.isNotEmpty(topicVoList)) {
                    if (topicVoList.size() > 6) {
                        topicVoList = topicVoList.stream().limit(6).collect(Collectors.toList());
                    }

                    Integer order = redisHashMapAdapter.get(RedisConfig.USER_READ_RECOMMEND_TOPIC.copy(), param.getUserId() + "", Integer.class);
                    if (null == order || ++order >= topicVoList.size()) {
                        order = 0;
                    }
                    redisHashMapAdapter.put(RedisConfig.USER_READ_RECOMMEND_TOPIC.copy(), param.getUserId() + "", order);
                    TopicVo topic = topicVoList.get(order);

                    List<News4Client> newList = new ArrayList<>();
                    for (int i = 0; i < list.size(); i++) {
                        if (i == 5) {
                            News4Client news4Client = new News4Client();
                            news4Client.setRecommendTopicVo(topic);
                            newList.add(news4Client);
                        }

                        newList.add(list.get(i));
                    }

                    result.getResult().setList(newList);
                }
            }
        }

        return result;
    }

    /**
     * 是否处于提包状态
     *
     * @param basicParam
     * @return
     */
    private Boolean getPublishState(BasicParam basicParam) {
        return appVersionIntegrationService.getPublishState(basicParam) && basicParam.getPlatform() == 2;
    }

    /**
     * 判断是否走2.1.0新版首页推荐
     * 如果满足地区已经开通社区，且频道为36或空，则走新的混合推荐器
     *
     * @return
     */
    private boolean checkIsNewVersionRecommend(HomeRecommendParam param) {
        return checkForumPostEnable(param.getAreaCode()) && (param.getKindId() == null || param.getKindId() == this.newsProperties.getRecommendKindId());
    }

    /**
     * 根据地区编号去查询该编号所在的地区是否开启社区
     * 旧版本不传areaCode字段，在此判断
     *
     * @param areaCode
     * @return
     */
    private boolean checkForumPostEnable(String areaCode) {
        if (com.bxm.newidea.component.tools.StringUtils.isNotEmpty(areaCode)) {
            LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(areaCode);
            if (locationDTO != null) {
                return locationDTO.getEnableCommunityContent() == 1;
            }
        }
        return false;
    }

    /**
     * 获得混合推荐返回类
     *
     * @param param
     * @param basicParam
     * @return
     */
    public Json<NewsMeta> listMixRecommendByPromotion(HomeRecommendParam param, BasicParam basicParam) {
        logger.debug("[listMixRecommendByPromotion]新闻推荐请求+1：{}", param);
        int count = 0;
        return retryMixResult(param, basicParam, count);
    }

    private Json<NewsMeta> retryMixResult(HomeRecommendParam param, BasicParam basicParam, Integer count) {
        long begin = System.currentTimeMillis();
        NewsMeta meta = new NewsMeta();

        MixRecommendParam mixRecommendParam = new MixRecommendParam();
        BeanUtils.copyProperties(param, mixRecommendParam);
        mixRecommendParam.setActionType(param.getRecommendType());
        mixRecommendParam.setPlatform(basicParam.getPlatform());

        //推荐器获取混合id列表
        List<MixRecommendDTO> mixRecommendDTOList = newsRecommendIntegrationService.listMixRecommendDTO(mixRecommendParam);
        logger.debug("[listMixRecommendByPromotion]从es中获取新闻列表信息,结果：{},耗时:{}", JSONObject.toJSONString(mixRecommendDTOList), System.currentTimeMillis() - begin);

        if (CollectionUtils.isEmpty(mixRecommendDTOList)) {
            return ResultUtil.genSuccessResult(meta);
        }

        //添加新闻和帖子
        List<News4Client> news4ClientList = listNews4Client(mixRecommendDTOList, param.getUserId(), param.getAreaCode(), true);

        //添加小视频(如果列表不为空且请求方不为小程序)
        if (CollectionUtils.isNotEmpty(news4ClientList) && 5 != basicParam.getPlatform()) {
            addRecommendVideo(param.getUserId(), param.getPageNum(), news4ClientList);
            logger.debug("[listMixRecommendByPromotion]小视频添加,耗时:{}", System.currentTimeMillis() - begin);
        }

        //隐藏敏感信息
        hideInfo(basicParam, meta);

        meta.setList(news4ClientList);
        meta.setNewsCount(meta.getList().size());

        logger.debug("[listMixRecommendByPromotion]拉取新闻列表完毕,结果:{},耗时：{}", JSONObject.toJSONString(meta), System.currentTimeMillis() - begin);
        return ResultUtil.genSuccessResult(meta);
    }

    @Override
    public List<News4Client> listNews4Client(List<MixRecommendDTO> mixRecommendDTOList, Long userId, String areaCode, boolean isFillTitle) {
        //分别得到新闻和帖子的id列表
        List<Long> postIdList = mixRecommendDTOList.stream().filter(mix -> "1".equals(mix.getOrigin())).map(MixRecommendDTO::getId).collect(Collectors.toList());
        Long[] newsIds = mixRecommendDTOList.stream().filter(mix -> !"1".equals(mix.getOrigin())).map(MixRecommendDTO::getId).toArray(Long[]::new);

        //根据id列表获得新闻列表内容，进行包装
        List<News> list = newsSearchIntegrationService.listRecommendedNews(newsIds);
        newsService.fillExtInfo(userId,list,areaCode);

        //根据id列表获得帖子列表内容，进行包装
        List<ForumPostVo> forumPostVoList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(postIdList)) {
            forumPostVoList = forumPostMapper.listPostByIdsInHome(postIdList);
            forumPostService.fillExtInfo(forumPostVoList, userId, areaCode, isFillTitle,(byte)1);
        }
        List<ForumPostDTO> forumPostResult = forumPostVoList.stream().map(ForumPostConvert::convertForumPost2DTO).collect(Collectors.toList());

        //包装返回实体对象
        List<News4Client> news4ClientList = new ArrayList<>();
        for (MixRecommendDTO mixRecommendDTO : mixRecommendDTOList) {
            // 如果是帖子
            if ("1".equals(mixRecommendDTO.getOrigin())) {
                ForumPostDTO post = forumPostResult.stream().filter(forumPostDTO -> forumPostDTO.getId().equals(mixRecommendDTO.getId())).findFirst().orElse(null);
                if (Objects.nonNull(post)) {
                    News4Client news4Client = new News4Client();
                    news4Client.setPost(post);
                    news4ClientList.add(news4Client);
                }
            } else {
                News news = list.stream().filter(newsDTO -> newsDTO.getId().equals(mixRecommendDTO.getId())).findFirst().orElse(null);
                if (Objects.nonNull(news) && 1 == news.getStatus()) {
                    News4Client news4Client = new News4Client();
                    news4Client.setNews(news);
                    news4ClientList.add(news4Client);
                }
            }
        }
        return news4ClientList;
    }

    @Override
    public Json<NewsMeta> listPostByMixRecommendPool(HomeRecommendParam param) {
        NewsMeta meta = new NewsMeta();
        List<Long> postIdList = mixedRecommendPoolMapper.listPostRecommend(param);
        if (CollectionUtils.isNotEmpty(postIdList)) {
            List<ForumPostVo> forumPostVoList = forumPostMapper.listPostByIdsInHome(postIdList);
            forumPostService.fillExtInfo(forumPostVoList, param.getUserId(), param.getAreaCode(), false,(byte)1);

            List<ForumPostDTO> forumPostResult = forumPostVoList.stream().map(ForumPostConvert::convertForumPost2DTO).collect(Collectors.toList());

            //包装返回实体对象
            List<News4Client> news4ClientList = new ArrayList<>();
            for (Long mixRecommendDTO : postIdList) {

                ForumPostDTO post = forumPostResult.stream().filter(forumPostDTO -> forumPostDTO.getId().equals(mixRecommendDTO)).findFirst().orElse(null);
                if (Objects.nonNull(post)) {
                    News4Client news4Client = new News4Client();
                    news4Client.setPost(post);
                    news4ClientList.add(news4Client);
                }

            }
            meta.setList(news4ClientList);
            meta.setNewsCount(meta.getList().size());
        }

        return ResultUtil.genSuccessResult(meta);
    }

}
