package com.bxm.localnews.news.service;

import com.bxm.component.mybatis.utils.BatchHelper;
import com.bxm.localnews.news.domain.UserTagConsumeMapper;
import com.bxm.localnews.news.vo.UserTag;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;

import static com.bxm.localnews.common.constant.RedisConfig.NEWS_RECOMMEND_INTERVAL;

@Service
public class WeightService extends BaseService {

    private static final long INTERVAL = 300000L;

    @Resource
    private UserTagConsumeMapper userTagConsumeMapper;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    /**
     * 重新计算用户的标签权重
     * @param userId 用户ID
     */
    public void recalcuUserTagWeight(Long userId) {
        //判断五分钟前是否进行过推荐
        Long lastRecommendTime = redisHashMapAdapter.get(NEWS_RECOMMEND_INTERVAL, userId.toString(), Long.class);

        long current = System.currentTimeMillis();
        if (lastRecommendTime == null || (current - lastRecommendTime > INTERVAL)) {
            redisHashMapAdapter.put(NEWS_RECOMMEND_INTERVAL, userId.toString(), current);
            execRecalcu(userId);
        }
    }

    private void execRecalcu(Long userId) {
        //获取用户的新标签
        List<UserTag> recountTags = userTagConsumeMapper.recountUserTags(userId);
        //获取用户的kind标签
        List<UserTag> kindTags = userTagConsumeMapper.getKindTags();

        for (UserTag kindTag : kindTags) {
            if (recountTags.stream().noneMatch(tag -> StringUtils.equals(kindTag.getName(), tag.getName()))) {
                kindTag.setUserId(userId);
                recountTags.add(kindTag);
            }
        }

        boolean removeEmpty = recountTags.removeIf(tag -> tag.getName() == null || "".equals(tag.getName()));
        if (removeEmpty) {
            logger.error("[{}]推荐的新闻标签包含空标签", userId);
        }

        //删除用户标签
        userTagConsumeMapper.deleteUserTag(userId);
        //批量新增用户标签
        batchSave(recountTags);
    }

    private void batchSave(List<UserTag> tags) {
        Date current = new Date();
        new BatchHelper<UserTagConsumeMapper, UserTag>(UserTagConsumeMapper.class, tags) {

            @Override
            protected int invoke(UserTag element) {
                element.setId(nextId());
                element.setUpdateTime(current);
                return this.mapper.insert(element);
            }
        };
    }

}
