package com.bxm.localnews.news.service.impl;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.alibaba.fastjson.JSONArray;
import com.bxm.localnews.base.service.AppVersionSupplyService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.news.constant.SensitiveFlagEnum;
import com.bxm.localnews.news.domain.NewsKindMapper;
import com.bxm.localnews.news.domain.UserKindMapper;
import com.bxm.localnews.news.dto.KindDTO;
import com.bxm.localnews.news.service.NewsKindService;
import com.bxm.localnews.news.vo.NewsKind;
import com.bxm.localnews.news.vo.UserKind;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import static com.bxm.localnews.common.constant.RedisConfig.NEWS_KIND;

/**
 * Created by mars on 2018/3/12.
 */
@Service("newsKindService")
@RefreshScope
public class NewsKindServiceImpl implements NewsKindService {

    private final static Logger logger = LoggerFactory.getLogger(NewsKindServiceImpl.class);

    private NewsKindMapper newsKindMapper;

    private UserKindMapper userKindMapper;

    private RedisStringAdapter redisStringAdapter;

    private RedisHashMapAdapter redisHashMapAdapter;

    private RedisSetAdapter redisSetAdapter;

    private AppVersionSupplyService appVersionSupplyService;

    @Autowired
    public NewsKindServiceImpl(NewsKindMapper newsKindMapper, UserKindMapper userKindMapper,
                               RedisStringAdapter redisStringAdapter, RedisHashMapAdapter redisHashMapAdapter,
                               RedisSetAdapter redisSetAdapter, AppVersionSupplyService appVersionSupplyService) {
        this.newsKindMapper = newsKindMapper;
        this.userKindMapper = userKindMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.redisSetAdapter = redisSetAdapter;
        this.appVersionSupplyService = appVersionSupplyService;
    }

    /**
     * 是否处于提包状态
     *
     * @param basicParam
     * @return
     */
    private Boolean getPublishState(BasicParam basicParam) {
        return appVersionSupplyService.getPublishState(basicParam);
    }

    @Override
    public List<NewsKind> selectAll(BasicParam basicParam) {
        List<NewsKind> allKinds = this.selectAllToRedis();
        if (getPublishState(basicParam)) {
            allKinds.removeIf(newsKind -> SensitiveFlagEnum.IS_SENSITIVE.getState().equals(newsKind.getSensitiveFlag()));
        }
        return allKinds;
    }

    @Override
    public Message createUserDefaultKinds(Long userId) {
        return Message.build(this.newsKindMapper.initUserDefaultKinds(userId));
    }

    @Override
    public Message createUserDefaultKindsForPublish(Long userId, BasicParam basicParam) {
        List<Integer> ids = Lists.newArrayList();
        if (appVersionSupplyService.getPublishState(basicParam)) {
            List<NewsKind> allKinds = this.selectAllToRedis();
            ids = allKinds.stream().filter(newsKind -> newsKind.getIsDefault() == 1).map(NewsKind::getId).collect(Collectors.toList());
        }

        List<Integer> userKindList = getFixKindId();
        if (CollectionUtils.isNotEmpty(ids)) {
            userKindList.addAll(ids);
        }
        userKindList = userKindList.stream().distinct().collect(Collectors.toList());

        List<UserKind> userKinds = Lists.newArrayList();
        for (int i = 0; i < userKindList.size(); i++) {
            UserKind userKind = new UserKind();
            userKind.setUserId(userId);
            userKind.setKindId(userKindList.get(i));
            userKind.setSortNo(i);
            userKinds.add(userKind);
        }

        userKindMapper.batchInsert(userKinds);
        return Message.build(true);
    }

    @Override
    public KindDTO getMyKindAndAllKind(Long userId, BasicParam basicParam) {
        List<NewsKind> allKinds = selectAll(basicParam);
        KindDTO kindDTO = new KindDTO(allKinds, null);
        if (null != userId) {
            List<NewsKind> myKinds = getMyKind(userId);

            if (getPublishState(basicParam)) {
                myKinds.removeIf(newsKind -> SensitiveFlagEnum.IS_SENSITIVE.getState().equals(newsKind.getSensitiveFlag()));
            }

            kindDTO = new KindDTO(allKinds, myKinds);
        }
        return kindDTO;
    }

    @Override
    public Message addUserKind(Long userId, Integer kindId) {
        List<NewsKind> myKinds = getMyKind(userId);
        if (myKinds.stream().filter(item -> item.getId().equals(kindId)).findAny().isPresent()) {
            return Message.build(false, "重复添加");
        }

        List<NewsKind> allKinds = selectAllToRedis();
        Optional<NewsKind> newsKindOptional = allKinds.stream().filter(item -> item.getId().equals(kindId)).findFirst();
        if (newsKindOptional.isPresent()) {
            NewsKind newsKind = newsKindOptional.get();
            myKinds.add(newsKind);
        }

        syncMyKind(myKinds, userId);
        return Message.build();
    }

    @Override
    public void resortUserKind(Long userId, Integer[] kindIds, String firstOpenType) {
        if (ArrayUtils.isEmpty(kindIds)) {
            return;
        }

        List<Integer> userKindIds = Stream.of(kindIds).collect(Collectors.toList());
        List<Integer> userKindList = getFixKindId();
        userKindList.addAll(userKindIds);
        userKindList = userKindList.stream().distinct().collect(Collectors.toList());

        List<NewsKind> allKinds = selectAllToRedis();
        List<NewsKind> myKinds = Lists.newArrayList();
        userKindList.forEach(id -> {
            allKinds.forEach(newsKind -> {
                if (newsKind.getId().equals(id)) {
                    myKinds.add(newsKind);
                }
            });
        });

        syncMyKind(myKinds, userId);
    }

    /**
     * 获取用户频道信息
     *
     * @param userId
     * @return
     */
    private List<NewsKind> getMyKind(Long userId) {
        TypeReference<List<NewsKind>> typeReference = new TypeReference<List<NewsKind>>() {
        };
        List<NewsKind> myKinds = this.redisHashMapAdapter.get(RedisConfig.NEWS_USER_KIND, userId.toString(), typeReference);

        if (CollectionUtils.isEmpty(myKinds)) {
            myKinds = this.userKindMapper.myKinds(userId);
            syncMyKind(myKinds, userId);
        }

        return myKinds;
    }

    /**
     * 更新redis的用户频道数据
     *
     * @param myKinds
     * @param userId
     */
    private void syncMyKind(List<NewsKind> myKinds, Long userId) {
        if (CollectionUtils.isNotEmpty(myKinds)) {
            this.redisHashMapAdapter.put(RedisConfig.NEWS_USER_KIND, userId.toString(), myKinds);
            //用户id存到set里---调度任务把数据同步到数据库
            this.redisSetAdapter.add(RedisConfig.NEWS_USER_KIND_SET, userId);
        }
    }

    /**
     * 获取所有的频道信息
     *
     * @return
     */
    private List<NewsKind> selectAllToRedis() {
        String str = this.redisStringAdapter.get(NEWS_KIND, String.class);
        List<NewsKind> newsKindList;
        if (StringUtils.isNotEmpty(str)) {
            newsKindList = JSONArray.parseArray(str, NewsKind.class);
        } else {
            newsKindList = this.newsKindMapper.selectAll();
            try {
                if (null != newsKindList) {
                    this.redisStringAdapter.set(NEWS_KIND, JSONArray.toJSONString(newsKindList));
                }
            } catch (Exception e) {
                logger.error("newsKind set redis error");
            }
        }
        return newsKindList;
    }

    /**
     * 获取固定频道的id
     *
     * @return
     */
    private List<Integer> getFixKindId() {
        List<NewsKind> allKinds = this.selectAllToRedis();

        return allKinds.stream().filter(newsKind -> newsKind.getIsFix() == 1).map(NewsKind::getId).collect(Collectors.toList());
    }
}
