package com.bxm.localnews.news.service.impl;

import com.bxm.localnews.base.service.AppVersionSupplyService;
import com.bxm.localnews.common.config.NewsProperties;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.integration.feign.NewsRecommendFeignService;
import com.bxm.localnews.integration.feign.NewsSearchFeignService;
import com.bxm.localnews.news.domain.NewsMapper;
import com.bxm.localnews.news.domain.NewsPoolMapper;
import com.bxm.localnews.news.domain.NewsRecommendedMapper;
import com.bxm.localnews.news.dto.ESNewsContentDTO;
import com.bxm.localnews.news.dto.VideoDto;
import com.bxm.localnews.news.param.VideoQueryParam;
import com.bxm.localnews.news.service.RecommendService;
import com.bxm.localnews.news.service.VideoRecommendService;
import com.bxm.localnews.news.service.WeightService;
import com.bxm.localnews.news.vo.News;
import com.bxm.localnews.news.vo.News4Client;
import com.bxm.localnews.news.vo.NewsMeta;
import com.bxm.localnews.news.vo.NewsRecommendParam;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.service.BaseService;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

@Service("recommendService")
public class RecommendServiceImpl extends BaseService implements RecommendService {

    /**
     * 每页获取新闻数量上限
     */
    private static final int MAX_PAGE_SIZE = 20;

    /**
     * 用户缓存警戒值，达到此值就进行用户缓存的追加
     */
    private static final int CACHE_ALARM_VALUE = MAX_PAGE_SIZE * 3;

    private NewsMapper newsMapper;

    private NewsRecommendedMapper newsRecommendedMapper;

    private AppVersionSupplyService appVersionSupplyService;

    private NewsProperties newsProperties;

    private VideoRecommendService videoRecommendService;

    private RedisListAdapter redisListAdapter;

    private NewsPoolMapper newsPoolMapper;

    private WeightService weightService;

    private NewsRecommendFeignService newsRecommendFeignService;

    private NewsSearchFeignService newsSearchFeignService;

    @Autowired
    public RecommendServiceImpl(NewsMapper newsMapper,
                                NewsRecommendedMapper newsRecommendedMapper,
                                NewsProperties newsProperties,
                                AppVersionSupplyService appVersionSupplyService,
                                VideoRecommendService videoRecommendService,
                                RedisListAdapter redisListAdapter,
                                NewsPoolMapper newsPoolMapper,
                                WeightService weightService,
                                NewsRecommendFeignService newsRecommendFeignService,
                                NewsSearchFeignService newsSearchFeignService) {
        this.newsMapper = newsMapper;
        this.newsRecommendedMapper = newsRecommendedMapper;
        this.newsProperties = newsProperties;
        this.appVersionSupplyService = appVersionSupplyService;
        this.videoRecommendService = videoRecommendService;
        this.redisListAdapter = redisListAdapter;
        this.newsPoolMapper = newsPoolMapper;
        this.weightService = weightService;
        this.newsRecommendFeignService = newsRecommendFeignService;
        this.newsSearchFeignService = newsSearchFeignService;
    }

    @Override
    public Json<NewsMeta> execRecommend(NewsRecommendParam param) {
        //调用推荐系统提供的推荐服务
        ResponseEntity<List<Long>> responseEntity = newsRecommendFeignService.recommendList(param.getUserId(),
                param.getKindId() == newsProperties.getRecommendKindId() ? null : param.getKindId().intValue(),
                param.getPagesize(), param.getKindId() == newsProperties.getRecommendKindId() ? param.getAreaCode() : null);
        List<Long> idList = responseEntity.getBody();
        logger.info("推荐引擎返回:{}",responseEntity.getBody());
        if(idList.isEmpty()){
            return ResultUtil.genFailedResult("暂时没有新闻了哦");
        }
        //获取新闻信息
        List<News> list = newsMapper.findNewsByIds(idList);
        Long[] ids = list.stream().mapToLong(News::getId).boxed().toArray(Long[]::new);
        //获取新闻内容
        ResponseEntity<List<ESNewsContentDTO>> newsRspEntity = newsSearchFeignService.multipleGet(ids);
        List<ESNewsContentDTO> dtos = newsRspEntity.getBody();
        for (News news : list) {
            if (dtos != null && !dtos.isEmpty()) {
                Optional<ESNewsContentDTO> optional = dtos.stream().filter(dto -> dto.getId().longValue() == news.getId().longValue()).findFirst();
                news.setContent(optional.isPresent() ? optional.get().getContent() : null);
            }
        }
        List<News> result = new ArrayList<>();
        for (Long id : idList) {
            Optional<News> optionalNews = list.stream().filter(news -> news.getId().longValue() == id.longValue()).findFirst();
            if (optionalNews.isPresent()) {
                result.add(optionalNews.get());
            }
        }

        NewsMeta meta = new NewsMeta();
        List<News4Client> news4ClientList = result.stream().map(news -> new News4Client(news, null)).collect(Collectors.toList());
        //添加小视频
        if (param.getKindId() == null || this.newsProperties.getRecommendKindId() == param.getKindId()) {
            addRecommendVideo(param, news4ClientList);
        }

        meta.setList(news4ClientList);
        meta.setNewsCount(meta.getList().size());

        //隐藏敏感信息
        hideInfo(param, meta);

        logger.info("推荐完成：{}", meta);
        return ResultUtil.genSuccessResult(meta);
    }

    /**
     * 推荐小视频，每2页显示一次小视频
     * @param param           推荐参数
     * @param news4ClientList 推荐返回结果元对象
     */
    private void addRecommendVideo(NewsRecommendParam param, List<News4Client> news4ClientList) {
        logger.info("NewsRecommendParam：{}", param);
        if (param.getCurPage() != 0 && param.getCurPage() % 2 == 0) {
            VideoQueryParam videoParam = new VideoQueryParam();
            videoParam.setUserId(param.getUserId());
            //在新闻列表中间插入
            List<VideoDto> videoDtoList = videoRecommendService.execRecommend(videoParam);
            if (CollectionUtils.isNotEmpty(videoDtoList)) {
                int size = news4ClientList.size();
                news4ClientList.add(size / 2, new News4Client(videoDtoList));
            }
        }
    }

    /**
     * 处于提包状态时，隐藏部分敏感信息，用于上包
     * 去除作者信息
     * @param param 推荐参数
     * @param meta  返回结果
     */
    private void hideInfo(NewsRecommendParam param, NewsMeta meta) {
        if (appVersionSupplyService.getPublishState(param)) {
            meta.getList().removeIf(client -> client.getVideos() != null);
            for (News4Client client : meta.getList()) {
                if (client.getNews() != null) {
                    client.getNews().setAuthor(null);
                    client.getNews().setAuthorImg(null);
                }
            }
        }
    }

}
