package com.bxm.localnews.news.service.impl;

import com.bxm.component.mybatis.utils.BatchHelper;
import com.bxm.localnews.integration.feign.NewsRecommendFeignService;
import com.bxm.localnews.news.domain.VideoMapper;
import com.bxm.localnews.news.domain.VideoRecommendMapper;
import com.bxm.localnews.news.dto.VideoDto;
import com.bxm.localnews.news.param.VideoQueryParam;
import com.bxm.localnews.news.service.VideoRecommendService;
import com.bxm.localnews.news.vo.VideoRecommend;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.List;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

import static com.bxm.localnews.common.constant.RedisConfig.VIDEO_RECOMMEND;

@Service
public class VideoRecommendServiceImpl extends BaseService implements VideoRecommendService {

    /**
     * 每页获取视频数量上限
     */
    private static final int MAX_PAGE_SIZE = 12;

    /**
     * 用户缓存警戒值，达到此值就进行用户缓存的追加
     */
    private static final int CACHE_ALARM_VALUE = MAX_PAGE_SIZE * 2;

    private VideoMapper videoMapper;

    private RedisListAdapter redisListAdapter;

    private VideoRecommendMapper videoRecommendMapper;

    /**
     * 触发用户推荐的计算信号量
     */
    private Condition waitCondition;

    private ReentrantLock lock;

    private NewsRecommendFeignService newsRecommendFeignService;

    @Autowired
    public VideoRecommendServiceImpl(VideoMapper videoMapper, RedisListAdapter redisListAdapter, VideoRecommendMapper videoRecommendMapper, NewsRecommendFeignService newsRecommendFeignService) {
        this.videoMapper = videoMapper;
        this.redisListAdapter = redisListAdapter;
        this.videoRecommendMapper = videoRecommendMapper;
        this.newsRecommendFeignService = newsRecommendFeignService;
    }

    @Override
    public List<VideoDto> execRecommend(VideoQueryParam param) {
        ResponseEntity<List<VideoDto>> responseEntity = newsRecommendFeignService.recommendVideoList(param.getUserId(), param.getPageSize());
        List<VideoDto> result = responseEntity.getBody();
        //将小视频的URL中的特殊字符进行转码
        if (CollectionUtils.isNotEmpty(result)) {
            result.forEach(video -> {
                String videoUrl = video.getVideoUrl();

                int fistQuestionMarkIndex = StringUtils.indexOf(videoUrl, "?") + 1;
                String firstPart = StringUtils.substring(videoUrl, 0, fistQuestionMarkIndex);

                videoUrl = StringUtils.substring(videoUrl, fistQuestionMarkIndex);
                try {
                    String queryPart = URLEncoder.encode(videoUrl, StandardCharsets.UTF_8.name());
                    queryPart = StringUtils.replace(queryPart, "%25", "%");
                    queryPart = StringUtils.replace(queryPart, "%26", "&");
                    queryPart = StringUtils.replace(queryPart, "%3D", "=");

                    video.setVideoUrl(firstPart + queryPart);
                } catch (UnsupportedEncodingException e) {
                    logger.error(e.getMessage(), e);
                }
            });
        }

        return result;
    }

    private void markRecommended(List<VideoDto> result, long userId) {
        //设置取出的视频为已推荐状态
        if (CollectionUtils.isNotEmpty(result)) {
            new BatchHelper<VideoRecommendMapper, VideoDto>(VideoRecommendMapper.class, result) {
                @Override
                protected int invoke(VideoDto element) {
                    VideoRecommend recommend = new VideoRecommend();
                    recommend.setAddTime(new Date());
                    recommend.setId(nextId());
                    recommend.setType((byte) 1);
                    recommend.setUserId(userId);
                    recommend.setVideoId(element.getId());
                    return videoRecommendMapper.insert(recommend);
                }
            };
        }
    }

    /**
     * 获取当前接口的实例，执行异步推荐
     * @param userId 用户ID
     */
    private void callAsyncRecommend(long userId) {
        VideoRecommendService recommendService = SpringContextHolder.getBean(VideoRecommendService.class);
        recommendService.doAsyncVideoRecommend(userId);
    }

    @Override
    @Async
    public void doAsyncVideoRecommend(long userId) {
        logger.debug("进入异步调用");
        long current = System.currentTimeMillis();
        //进行推荐
        KeyGenerator userCacheKey = VIDEO_RECOMMEND.copy().appendKey(userId);
        List<VideoDto> result = videoMapper.getUserRecommend(userId);

        VideoDto[] videoDtos = result.toArray(new VideoDto[0]);
        //移除剩余的缓存数据，防止重复数据
        redisListAdapter.remove(userCacheKey);
        redisListAdapter.rightPush(userCacheKey, videoDtos);
        //保存30分钟
        redisListAdapter.expire(userCacheKey, 1800);

        logger.debug("给[{}]进行小视频推荐。花费时间[{}]", userId, System.currentTimeMillis() - current);
    }

}


















