package com.bxm.localnews.payment.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alipay.api.AlipayApiException;
import com.alipay.api.internal.util.AlipaySignature;
import com.bxm.localnews.payment.config.AlipayProperties;
import com.bxm.localnews.payment.constant.PayTypeEnum;
import com.bxm.localnews.payment.order.PaymentOrderFactory;
import com.bxm.localnews.payment.pay.PayProxyService;
import com.bxm.localnews.payment.request.AlipayNotifyResult;
import com.bxm.newidea.component.controller.BaseController;
import com.bxm.newidea.component.util.WebUtils;
import com.github.binarywang.wxpay.bean.notify.WxPayNotifyResponse;
import com.google.common.collect.Maps;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import springfox.documentation.annotations.ApiIgnore;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;
import java.util.Map;

/**
 * @author zhaoyadong 2018/12/27 20:16
 * @desc
 */
@Api(tags = "6-02 支付回调相关接口", description = "包括支付回调接口操作")
@ApiIgnore
@RestController
@RequestMapping("/api/notify")

public class PayNotifyController extends BaseController {

    private PayProxyService payProxyService;

    private AlipayProperties alipayProperties;

    private final PaymentOrderFactory paymentOrderFactory;

    @Autowired
    public PayNotifyController(PayProxyService payProxyService, AlipayProperties alipayProperties,PaymentOrderFactory paymentOrderFactory) {
        this.payProxyService = payProxyService;
        this.alipayProperties = alipayProperties;
        this.paymentOrderFactory = paymentOrderFactory;
    }

    @ApiOperation(value = "微信支付回调通知处理", notes = "")
    @PostMapping("/wechat")
    public String parseOrderNotifyResult(@RequestBody String xmlData) {
        String result = payProxyService.notifyPay(xmlData, PayTypeEnum.WX_PAY);
        if (result == null) {
            return WxPayNotifyResponse.fail("FAIL");
        }
        return WxPayNotifyResponse.success("SUCCESS");
    }

    @ApiOperation(value = "支付宝回调通知处理", notes = "")
    @PostMapping("/alipay")
    public void aliPayNotifyResult(HttpServletRequest request, HttpServletResponse response) {
        try {
            PrintWriter writer = response.getWriter();

            if (alipayProperties.isEnableCallbackValid() && !verifyResult(request)) {
                logger.info("支付宝回调通知验证失败,支付回调结果：{}", WebUtils.getReqeustParam(request));
                writer.print("fail");
                return;
            }

            AlipayNotifyResult result = parseReqeust(request);
            String jsonResult = JSON.toJSONString(result);

            logger.info("支付宝回调结果：{}", jsonResult);

            payProxyService.notifyPay(jsonResult, PayTypeEnum.ALI_PAY);

            writer.print("success");
        } catch (AlipayApiException e) {
            logger.error("支付宝回调通知：数据解析异常: {}", e);
        } catch (IOException ioe) {
            logger.error("response获取writer异常:{}", ioe);
        }
    }

    @ApiOperation(value = "支付宝回调通知处理(预发布环境支持用)")
    @PostMapping("pre/alipay")
    public void alipayNodifyPreResult(HttpServletRequest request, HttpServletResponse response) {
        aliPayNotifyResult(request, response);
    }

    private AlipayNotifyResult parseReqeust(HttpServletRequest request) {
        return AlipayNotifyResult.builder()
                .outTradeNo(getField("out_trade_no", request))
                .tradeStatus(getField("trade_status", request))
                .notifyTime(getField("notify_time", request))
                .tradeNo(getField("trade_no", request))
                .build();
    }

    private String getField(String fieldName, HttpServletRequest request) {
        return new String(request.getParameter(fieldName)
                .getBytes(StandardCharsets.ISO_8859_1), StandardCharsets.UTF_8);
    }

    /**
     * request提取参数
     * @param request
     * @return
     */
    private Map<String, String> covertMap(HttpServletRequest request) {
        Map<String, String> map = Maps.newHashMap();
        Map requestParams = request.getParameterMap();
        for (Iterator iter = requestParams.keySet().iterator(); iter.hasNext(); ) {
            String name = (String) iter.next();
            String[] values = (String[]) requestParams.get(name);
            String valueStr = "";
            for (int i = 0; i < values.length; i++) {
                valueStr = (i == values.length - 1) ? valueStr + values[i]
                        : valueStr + values[i] + ",";
            }
            map.put(name, valueStr);
        }
        logger.info("支付宝异步通知返回信息：{}", JSONObject.toJSONString(map));
        return map;
    }

    /**
     * 支付宝通知验证结果
     * @param request
     * @return
     */
    private Boolean verifyResult(HttpServletRequest request) throws AlipayApiException {
        return AlipaySignature.rsaCheckV1(covertMap(request),
                alipayProperties.getAlipayPublicKey(),
                alipayProperties.getCharset(),
                alipayProperties.getSigntype());
    }

    /**
     * 微信退款结果回调通知
     */
    @RequestMapping(value = "/wx/refund", method = RequestMethod.POST)
    public String refund(@RequestBody String xml) {
        return paymentOrderFactory.execRefundCallback(xml, PayTypeEnum.WX_PAY);
    }
}
