package com.bxm.localnews.payment.pay.alipay;

import com.alipay.api.AlipayApiException;
import com.alipay.api.AlipayClient;
import com.alipay.api.AlipayResponse;
import com.alipay.api.domain.AlipayTradeQueryModel;
import com.alipay.api.domain.AlipayTradeRefundModel;
import com.alipay.api.domain.AlipayTradeWapPayModel;
import com.alipay.api.request.AlipayTradeQueryRequest;
import com.alipay.api.request.AlipayTradeRefundRequest;
import com.alipay.api.request.AlipayTradeWapPayRequest;
import com.alipay.api.response.AlipayTradeQueryResponse;
import com.alipay.api.response.AlipayTradeRefundResponse;
import com.bxm.localnews.base.service.BaseUrlFacadeService;
import com.bxm.localnews.payment.config.AlipayProperties;
import com.bxm.localnews.payment.config.PayProperties;
import com.bxm.localnews.payment.constant.PayTypeEnum;
import com.bxm.localnews.payment.constant.PaymentStatusEnum;
import com.bxm.localnews.payment.dto.PayCallBackResult;
import com.bxm.localnews.payment.param.PaymentRefundParam;
import com.bxm.localnews.payment.pay.impl.AbstractPayModeService;
import com.bxm.localnews.payment.request.AlipayNotifyResult;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.localnews.payment.vo.PaymentOrderDetail;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.NumberUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.Objects;

import static com.alibaba.fastjson.JSON.parseObject;
import static com.alibaba.fastjson.JSON.toJSONString;
import static java.math.RoundingMode.DOWN;
import static java.math.RoundingMode.HALF_DOWN;
import static java.util.Objects.isNull;
import static org.apache.commons.lang3.StringUtils.*;

@Component
@Slf4j
@AllArgsConstructor
public class AliPayModeServiceImpl extends AbstractPayModeService {

    private PayProperties payProperties;

    private AlipayProperties alipayProperties;

    private AlipayClient alipayClient;

    private BaseUrlFacadeService baseUrlFacadeService;

    private static final String SUCCESS_CODE = "10000";

    @Override
    public PayTypeEnum support() {
        return PayTypeEnum.ALI_PAY;
    }

    @Override
    public void create(PaymentOrderDetail paymentOrderDetail) {
        try {
            AlipayResponse alipayResponse = alipayClient.pageExecute(createWapPayRequest(paymentOrderDetail));
            paymentOrderDetail.setLink(alipayResponse.getBody());
            logger.info("支付宝预支付返回信息：{}", alipayResponse.getBody());
        } catch (AlipayApiException e) {
            logger.error("用户id{}，订单id{}，创建支付宝预支付信息失败", paymentOrderDetail.getUserId(), paymentOrderDetail.getPaymentNum(), e);
        }
    }

    @Override
    public void query(PaymentOrder paymentOrder) {
        try {
            AlipayTradeQueryResponse alipayTradeQueryResponse = alipayClient.execute(createTradeQueryRequest(paymentOrder.getPaymentNum()));
            logger.info("支付宝查询订单返回的信息：{}", alipayTradeQueryResponse.getBody());
        } catch (AlipayApiException e) {
            logger.error("支付宝订单[{}]查询相关信息失败", paymentOrder.getPaymentNum(), e);
        }
    }

    @Override
    public PayCallBackResult doCallBack(String data) {
        AlipayNotifyResult result = parseObject(data, AlipayNotifyResult.class);

        if (isNull(result)) {
            logger.warn("支付宝回调信息: {} 有误 回调失败", data);
            return buildFailedResult();
        }

        PaymentOrder paymentOrder = payService.getPaymentOrderByPaymentNo(result.getOutTradeNo());
        if (paymentOrder == null) {
            logger.warn("接收到的支付宝支付回调参数错误,支付订单[{}]不存在", result.getOutTradeNo());
            return buildFailedResult();
        }

        // 除了支付成功的状态外，其他状态均更新为支付成功
        if (PaymentStatusEnum.SUCCEED.getType().equals(paymentOrder.getStatus())) {
            logger.warn("支付订单[{}]重复处理，状态为：[{}]", result.getOutTradeNo(), paymentOrder.getStatus());
            return buildFailedResult();
        }

        if(!PaymentStatusEnum.WAIT.getType().equals(paymentOrder.getStatus())){
            logger.warn("订单[{}]当前状态不是待付款，此情况是异常状态，请注意业务流程和逻辑", paymentOrder.getPaymentNum());
        }

        paymentOrder.setTradeNo(result.getTradeNo());
        paymentOrder.setFinishTime(DateUtils.parseDateNonStrict(result.getNotifyTime()));

        if (equalsAny(result.getTradeStatus(), "TRADE_FINISHED", "TRADE_SUCCESS")) {

            paymentOrder.setStatus(PaymentStatusEnum.SUCCEED.getType());
            payService.modifyStatus(paymentOrder);
        } else if ("TRADE_CLOSED".equals(result.getTradeStatus())) {

            paymentOrder.setStatus(PaymentStatusEnum.UNDO.getType());
            payService.modifyStatus(paymentOrder);
        }

        return buildSuccessResult(null, paymentOrder);
    }

    /**
     * 创建预支付信息
     * @param paymentOrderDetail 订单信息
     * @return
     */
    private AlipayTradeWapPayModel createWapPayModel(PaymentOrderDetail paymentOrderDetail) {
        if (log.isDebugEnabled()) {
            logger.debug("payProperties:{}", payProperties);
            logger.debug("payProperties.vipPrice:{}", payProperties.getVipPrice());
        }

        AlipayTradeWapPayModel model = new AlipayTradeWapPayModel();
        model.setOutTradeNo(paymentOrderDetail.getPaymentNum());
        model.setSubject(paymentOrderDetail.getPaymentTitle());
        if (Objects.nonNull(paymentOrderDetail.getAmount())) {
            model.setTotalAmount(Objects.toString(paymentOrderDetail.getAmount().setScale(2, DOWN)));
        }

        model.setBody(paymentOrderDetail.getPaymentTitle());
        model.setProductCode(alipayProperties.getProductCode());
        model.setTimeoutExpress(alipayProperties.getTimeoutExpress());
        return model;
    }

    /**
     * 创建支付请求
     * @param paymentOrderDetail
     * @return
     */
    private AlipayTradeWapPayRequest createWapPayRequest(PaymentOrderDetail paymentOrderDetail) {
        AlipayTradeWapPayRequest alipayRequest = new AlipayTradeWapPayRequest();
        alipayRequest.setBizModel(createWapPayModel(paymentOrderDetail));
        alipayRequest.setNotifyUrl(join(baseUrlFacadeService.getServerHostBaseUrl(), alipayProperties.getNotifyPath()));

        if (isNotBlank(paymentOrderDetail.getReturnUrl())) {
            StringBuilder returnUrl = new StringBuilder();

            if (!startsWith(paymentOrderDetail.getReturnUrl(), "http")) {
                returnUrl.append(baseUrlFacadeService.getInnerH5BaseUrl());
            }

            returnUrl.append(paymentOrderDetail.getReturnUrl());
            returnUrl.append("&paymentNum=");
            returnUrl.append("paymentOrderDetail.getPaymentNum()");

            alipayRequest.setReturnUrl(returnUrl.toString());
        }

        if (log.isDebugEnabled()) {
            log.debug("创建支付宝支付请求参数: {}", toJSONString(alipayRequest));
        }

        return alipayRequest;
    }

    /**
     * 查询订单信息
     * @param paymentNum
     * @return
     */
    private AlipayTradeQueryModel createTradeQueryModel(String paymentNum) {
        AlipayTradeQueryModel model = new AlipayTradeQueryModel();
        model.setOutTradeNo(paymentNum);
        return model;
    }

    /**
     * 创建查询订单请求
     * @param paymentNum
     * @return
     */
    private AlipayTradeQueryRequest createTradeQueryRequest(String paymentNum) {
        AlipayTradeQueryRequest alipayTradeQueryRequest = new AlipayTradeQueryRequest();
        alipayTradeQueryRequest.setBizModel(createTradeQueryModel(paymentNum));
        return alipayTradeQueryRequest;
    }

    @Override
    public Message submitRefund(PaymentRefundParam param) {
        log.info("发起支付宝退款申请,请求参数：{}", param);

        BigDecimal totalFee = param.getPayOrder().getAmount();
        BigDecimal refundFee;
        String outRefundNo = buildRefundOrderNo(param);

        if (Objects.equals(param.getFullAmount(), Boolean.TRUE)) {
            refundFee = totalFee;
        } else {
            refundFee = param.getRefundFee();
        }

        AlipayTradeRefundModel model = new AlipayTradeRefundModel();
        //退款对应的支付交易号
        model.setTradeNo(param.getPayOrder().getTradeNo());
        //退款订单号（我方定义）,多次退款需要保证退款订单号一致
        model.setOutRequestNo(outRefundNo);
        //退款金额，支持两位小数
        model.setRefundAmount(NumberUtils.format(refundFee.setScale(2, HALF_DOWN).floatValue()));
        model.setRefundReason("订单退款");

        AlipayTradeRefundRequest tradeRefundRequest = new AlipayTradeRefundRequest();
        tradeRefundRequest.setBizModel(model);

        try {
            createRefundOrder(param, outRefundNo, refundFee);
            if (log.isDebugEnabled()) {
                log.debug("tradeRefundRequest: {} ", toJSONString(tradeRefundRequest));
            }
            AlipayTradeRefundResponse response = alipayClient.execute(tradeRefundRequest);
            if (equalsAny(response.getCode(), SUCCESS_CODE)) {
                log.info("支付申请退款成功，请求参数：{}，返回结果：{}", param, toJSONString(response));

                changeRefundOrderStatus(outRefundNo,
                        response.getTradeNo(),
                        toJSONString(response),
                        PaymentStatusEnum.REFUND);
            } else {
                log.error("支付宝退款申请失败，请求参数：{}，请求结果：{}", param, toJSONString(response));

                changeRefundOrderStatus(outRefundNo,
                        response.getTradeNo(),
                        toJSONString(response),
                        PaymentStatusEnum.REFUND_FAIL);
                return Message.build(false, "退款申请失败");
            }
        } catch (AlipayApiException e) {
            logger.error(e.getMessage(), e);
            return Message.build(false, "退款申请异常");
        }

        logger.debug("支付退款申请处理完成,请求参数：{}", param);

        return Message.build(true);
    }

    @Override
    public String execRefundCallback(String callbackRequest) {
        throw new UnsupportedOperationException("支付宝无退款回调处理");
    }
    /**
     * 根据付款订单构建退款订单编号
     * @param param 退款订单创建参数
     * @return 退款订单编号
     */
    @Override
    protected String buildRefundOrderNo(PaymentRefundParam param) {
        return "R" + param.getPayOrder().getPaymentNum();
    }

}
