package com.bxm.localnews.payment.pay.wechat;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.base.service.BaseUrlFacadeService;
import com.bxm.localnews.payment.config.WxPayProperties;
import com.bxm.localnews.payment.constant.PaymentStatusEnum;
import com.bxm.localnews.payment.dto.PayCallBackResult;
import com.bxm.localnews.payment.param.PaymentRefundParam;
import com.bxm.localnews.payment.pay.PaymentOrderService;
import com.bxm.localnews.payment.pay.impl.AbstractPayModeService;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.Message;
import com.github.binarywang.wxpay.bean.notify.WxPayOrderNotifyResult;
import com.github.binarywang.wxpay.bean.notify.WxPayRefundNotifyResult;
import com.github.binarywang.wxpay.bean.request.BaseWxPayRequest;
import com.github.binarywang.wxpay.bean.request.WxPayRefundRequest;
import com.github.binarywang.wxpay.bean.result.WxPayOrderQueryResult;
import com.github.binarywang.wxpay.bean.result.WxPayRefundResult;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.binarywang.wxpay.service.WxPayService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.math.BigDecimal;
import java.math.RoundingMode;

import static org.apache.commons.lang3.StringUtils.join;

@Slf4j
public abstract class AbstractWechatPayModeService extends AbstractPayModeService {

    private String successCode = "SUCCESS";

    @Autowired
    protected WxWithdrawProxyService wxWithdrawProxyService;

    @Autowired
    protected WxPayService wxPayService;

    @Autowired
    protected PaymentOrderService payService;

    @Autowired
    private WxPayProperties properties;

    @Autowired
    private BaseUrlFacadeService baseUrlFacadeService;

    @Override
    public PayCallBackResult doCallBack(String data) {
        WxPayOrderNotifyResult result = wxWithdrawProxyService.parseOrderNotifyResult(data);
        String paymentNum = result.getOutTradeNo();
        PaymentOrder paymentOrder = payService.getPaymentOrderByPaymentNo(paymentNum);
        if (paymentOrder == null) {
            log.error("接收到的微信支付回调参数错误,支付订单[{}]不存在", paymentNum);
            return buildFailedResult();
        }

        Integer amount = BaseWxPayRequest.yuanToFen(paymentOrder.getAmount().toString());
        if (amount.intValue() != result.getTotalFee().intValue()) {
            log.error("支付订单支付金额[{}]与微信返回的订单总金额[{}]不匹配", amount, result.getTotalFee());
            return buildFailedResult();
        }

        if (!successCode.equals(result.getReturnCode()) || !successCode.equals(result.getResultCode())) {
            logger.error("订单[{}]回调结果为失败", paymentNum);
            return buildFailedResult();
        }

        // 如果订单状态已经是付款成功，则直接返回，防止重复处理
        if (PaymentStatusEnum.SUCCEED.getType().equals(paymentOrder.getStatus())) {
            return buildSuccessResult(successCode, paymentOrder);
        }

        //逻辑修正，支付订单如果状态不是支付成功，均执行支付成功的处理逻辑
        if (!PaymentStatusEnum.WAIT.getType().equals(paymentOrder.getStatus())) {
            logger.warn("订单[{}]当前状态不是待付款，此情况是异常状态，请注意业务流程和逻辑", paymentNum);
        }

        query(paymentOrder);

        return buildSuccessResult(successCode, paymentOrder);
    }

    @Override
    public void query(PaymentOrder paymentOrder) {
        //小程序暂时没有订单相关需求，如果需要则要在订单信息中保存对应的商户平台
        WxPayOrderQueryResult result = wxWithdrawProxyService.queryOrder(paymentOrder.getPaymentNum(), (byte) 1);

        if (!successCode.equals(result.getReturnCode()) ||
                !successCode.equals(result.getResultCode())) {
            // TODO 查询失败，但是回调成功，需要放到重试队列
            log.error("查询订单[{}]微信支付失败", paymentOrder.getPaymentNum());
            return;
        }

        if (successCode.equals(result.getTradeState())) {
            afterPaySucceed(result, paymentOrder);
        } else {
            afterPayClose(result, paymentOrder);
        }
    }

    /**
     * 取消支付回调函数
     *
     * @param result       包含回调请求的相关信息
     * @param paymentOrder 需要更新的支付订单
     */
    private void afterPayClose(WxPayOrderQueryResult result, PaymentOrder paymentOrder) {
        paymentOrder.setStatus(PaymentStatusEnum.UNDO.getType());
        paymentOrder.setFinishTime(DateUtils.parseDateNonStrict(result.getTimeEnd()));
        paymentOrder.setResult(result.getTradeState());
        paymentOrder.setTradeNo(result.getTransactionId());

        payService.modifyPaymentOrderStatus(paymentOrder);
    }

    /**
     * 确认支付成功后的回调函数
     *
     * @param result       包含回调请求的相关信息
     * @param paymentOrder 需要更新的支付订单
     */
    private void afterPaySucceed(WxPayOrderQueryResult result, PaymentOrder paymentOrder) {
        paymentOrder.setStatus(PaymentStatusEnum.SUCCEED.getType());
        paymentOrder.setFinishTime(DateUtils.parseDateNonStrict(result.getTimeEnd()));
        paymentOrder.setResult(result.getErrCodeDes());
        paymentOrder.setTradeNo(result.getTransactionId());

        payService.modifyPaymentOrderStatus(paymentOrder);
    }


    private String buildSuccessResponse() {
        return "<xml><return_code><![CDATA[SUCCESS]]></return_code><return_msg><![CDATA[OK]]></return_msg></xml>";
    }

    @Override
    public String execRefundCallback(String callbackRequest) {
        log.info("微信退款回调参数：{}", callbackRequest);

        try {
            WxPayRefundNotifyResult notifyResult = wxPayService.parseRefundNotifyResult(callbackRequest);

            if (StringUtils.equals(notifyResult.getReturnCode(), successCode)) {
                //判断业务状态是否成功
                if (StringUtils.equals(notifyResult.getReqInfo().getRefundStatus(), successCode)) {
                    changeRefundOrderStatus(notifyResult.getReqInfo().getOutTradeNo(),
                            notifyResult.getReqInfo().getOutRefundNo(),
                            JSON.toJSONString(notifyResult.getReqInfo()),
                            PaymentStatusEnum.REFUND);
                } else {
                    log.error("退款业务状态异常，回调参数：{}，回调结果：{}",
                            callbackRequest,
                            JSON.toJSONString(notifyResult));

                    changeRefundOrderStatus(notifyResult.getReqInfo().getOutTradeNo(),
                            notifyResult.getReqInfo().getOutRefundNo(),
                            JSON.toJSONString(notifyResult.getReqInfo()),
                            PaymentStatusEnum.REFUND_FAIL);
                }
            } else {
                log.error("微信退款失败，回调参数：{}，回调结果：{}",
                        callbackRequest,
                        JSON.toJSONString(notifyResult));
            }

            return buildSuccessResponse();
        } catch (WxPayException e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    @Override
    public Message submitRefund(PaymentRefundParam param) {
        log.debug("发起退款申请，请求参数：{}", param);

        WxPayRefundRequest request = new WxPayRefundRequest();
        try {
            BigDecimal multiple = new BigDecimal("100");
            BigDecimal totalFee = param.getPayOrder().getAmount().multiply(multiple);
            BigDecimal refundFee;
            String outRefundNo = buildRefundOrderNo(param);

            if (param.getFullAmount()) {
                refundFee = totalFee;
            } else {
                refundFee = param.getRefundFee().multiply(multiple);
            }

            // 订单流水号
            request.setTransactionId(param.getPayOrder().getTradeNo());
            //退款订单编号（我方）
            request.setOutRefundNo(outRefundNo);
            //退款的原始订单金额
            request.setTotalFee(totalFee.intValue());
            //退还金额
            request.setRefundFee(refundFee.intValue());
            //操作人（商户号）
            request.setOpUserId(wxPayService.getConfig().getMchId());
            //退款操作后的回调地址
            request.setNotifyUrl(join(baseUrlFacadeService.getServerHostBaseUrl(), properties.getRefundNotifyPath()));

            //校验参数并进行签名
            request.checkAndSign(wxPayService.getConfig());

            WxPayRefundResult refundResult = wxPayService.refund(request);

            if (StringUtils.equals(refundResult.getReturnCode(), successCode)) {
                log.info("申请退款成功，请求参数：{}，返回结果：{}", param, JSON.toJSONString(refundResult));
                //调用成功后，创建退款订单
                createRefundOrder(param, outRefundNo, refundFee.divide(multiple, RoundingMode.HALF_DOWN));

            } else {
                log.error("退款申请调用失败，请求参数：{}，返回结果：{}", param, JSON.toJSONString(refundResult));
                return Message.build(false, "微信支付调用失败");
            }

            log.debug("退款申请发起成功，请求参数：{}，返回值：{}", param, JSON.toJSONString(refundResult));
        } catch (WxPayException e) {
            logger.error(e.getMessage(), e);
            return Message.build(false, "调用微信支付平台退款API失败");
        }

        return Message.build(true);
    }

    /**
     * 根据付款订单构建退款订单编号
     *
     * @param param 退款订单创建参数
     * @return 退款订单编号
     */
    @Override
    protected String buildRefundOrderNo(PaymentRefundParam param) {
        return "R" + param.getPayOrder().getPaymentNum();
    }

}
