package com.bxm.localnews.payment.pay.wechat;

import com.bxm.localnews.payment.config.WxPayProperties;
import com.bxm.localnews.payment.request.WxPayQueryWithdrawRequest;
import com.bxm.localnews.payment.request.WxPayWithdrawRequest;
import com.bxm.localnews.payment.result.QueryWithdrawResult;
import com.bxm.localnews.payment.result.WxWithdrawResult;
import com.bxm.localnews.payment.vo.PaymentOrderDetail;
import com.bxm.localnews.thirdparty.constant.WechatMpTypeEnum;
import com.bxm.localnews.thirdparty.dto.WxUserInfo;
import com.bxm.localnews.thirdparty.service.WxMpFacadeService;
import com.github.binarywang.wxpay.bean.notify.WxPayOrderNotifyResult;
import com.github.binarywang.wxpay.bean.order.WxPayAppOrderResult;
import com.github.binarywang.wxpay.bean.order.WxPayMpOrderResult;
import com.github.binarywang.wxpay.bean.order.WxPayMwebOrderResult;
import com.github.binarywang.wxpay.bean.request.BaseWxPayRequest;
import com.github.binarywang.wxpay.bean.request.WxPayUnifiedOrderRequest;
import com.github.binarywang.wxpay.bean.result.WxPayOrderQueryResult;
import com.github.binarywang.wxpay.constant.WxPayConstants;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.binarywang.wxpay.service.WxPayService;
import com.github.binarywang.wxpay.service.impl.WxPayServiceImpl;
import com.thoughtworks.xstream.XStream;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 基于第三方实现的微信提现代理实现
 * @author zhaoyadong 2019/3/11 20:38
 */
@Component
@Slf4j
public class WxWithdrawProxyService extends WxPayServiceImpl {

    private final WxPayService wxPayService;

    private final WxPayService wxWithdrawPayService;

    private final WxPayService wxAppletPayService;

    private final WxPayProperties wxPayProperties;

    private final WxMpFacadeService wxMpFacadeService;

    @Autowired
    public WxWithdrawProxyService(WxPayService wxPayService,
                                  WxPayService wxWithdrawPayService,
                                  WxPayService wxAppletPayService,
                                  WxPayProperties wxPayProperties,
                                  WxMpFacadeService wxMpFacadeService) {
        this.wxPayService = wxPayService;
        this.wxWithdrawPayService = wxWithdrawPayService;
        this.wxAppletPayService = wxAppletPayService;
        this.wxPayProperties = wxPayProperties;
        this.wxMpFacadeService = wxMpFacadeService;
    }

    @Override
    public byte[] postForBytes(String url, String requestStr, boolean useKey) {
        return new byte[0];
    }

    @Override
    public String post(String url, String requestStr, boolean useKey) {
        return null;
    }

    private WxPayService getWxPayServiceByType(Byte withType) {
        if (null != withType && 2 == withType) {
            return wxAppletPayService;
        }
        return wxPayService;
    }

    /**
     * 提现
     */
    WxWithdrawResult transfers(WxPayWithdrawRequest request) throws WxPayException {
        request.checkAndSign(wxWithdrawPayService.getConfig());
        String url = wxWithdrawPayService.getPayBaseUrl() + "/mmpaymkttransfers/promotion/transfers";
        String responseContent = wxWithdrawPayService.post(url, request.toXML(), Boolean.TRUE);

        XStream xstream = new XStream();
        XStream.setupDefaultSecurity(xstream);
        xstream.processAnnotations(WxWithdrawResult.class);
        xstream.allowTypes(new Class[]{WxWithdrawResult.class});
        return (WxWithdrawResult) xstream.fromXML(responseContent);
    }

    /**
     * 查询企业付款
     */
    QueryWithdrawResult queryWithdraw(WxPayQueryWithdrawRequest request) throws WxPayException {
        request.checkAndSign(wxWithdrawPayService.getConfig());
        String url = this.getPayBaseUrl() + "/mmpaymkttransfers/gettransferinfo";
        String responseContent = super.post(url, request.toXML(), Boolean.TRUE);

        XStream xstream = new XStream();
        XStream.setupDefaultSecurity(xstream);
        xstream.processAnnotations(QueryWithdrawResult.class);
        xstream.allowTypes(new Class[]{QueryWithdrawResult.class});
        return (QueryWithdrawResult) xstream.fromXML(responseContent);
    }

    /**
     * 微信支付结果通知
     */
    @Override
    public WxPayOrderNotifyResult parseOrderNotifyResult(String data) {
        try {
            WxPayService wxPayService = getWxPayServiceByType(null);
            return wxPayService.parseOrderNotifyResult(data);
        } catch (WxPayException e) {
            log.error("微信支付回调异常:", e);
            return null;
        }
    }

    /**
     * 查询微信订单
     */
    WxPayOrderQueryResult queryOrder(String paymentNum, Byte withType) {
        try {
            WxPayService wxPayService = getWxPayServiceByType(withType);
            return wxPayService.queryOrder(null, paymentNum);
        } catch (WxPayException e) {
            log.error("查询订单[{}]异常: [{}]", paymentNum, e);
            return null;
        }
    }

    /**
     * APP下单返回信息(仅有小程序仍在使用)
     */
    WxPayAppOrderResult createAppOrder(PaymentOrderDetail paymentOrder) {
        try {
            WxPayUnifiedOrderRequest wxPayUnifiedOrderRequest = new WxPayUnifiedOrderRequest();

            wxPayUnifiedOrderRequest.setDeviceInfo("APP");
            wxPayUnifiedOrderRequest.setOutTradeNo(paymentOrder.getPaymentNum());
            wxPayUnifiedOrderRequest.setBody(wxPayProperties.getBody());
            wxPayUnifiedOrderRequest.setSceneInfo(wxPayProperties.getSceneInfo());
            wxPayUnifiedOrderRequest.setTotalFee(BaseWxPayRequest.yuanToFen(paymentOrder.getAmount().toString()));
            wxPayUnifiedOrderRequest.setSpbillCreateIp(paymentOrder.getClientIp());
            wxPayUnifiedOrderRequest.setTradeType(WxPayConstants.TradeType.APP);

            return wxAppletPayService.createOrder(wxPayUnifiedOrderRequest);

        } catch (WxPayException e) {
            log.error("微信下单失败！订单号：{},原因:{}", paymentOrder.getPaymentNum(), e.getMessage());
            log.error("下单异常：", e);
        }

        return null;
    }

    /**
     * H5支付下单
     * @param paymentOrder 订单信息
     * @return 预下单结果
     */
    WxPayMwebOrderResult createWebOrder(PaymentOrderDetail paymentOrder) {
        try {
            WxPayUnifiedOrderRequest wxPayUnifiedOrderRequest = new WxPayUnifiedOrderRequest();

            wxPayUnifiedOrderRequest.setDeviceInfo("WEB");
            wxPayUnifiedOrderRequest.setOutTradeNo(paymentOrder.getPaymentNum());
            wxPayUnifiedOrderRequest.setBody(wxPayProperties.getBody());
            wxPayUnifiedOrderRequest.setSceneInfo(wxPayProperties.getSceneInfo());
            wxPayUnifiedOrderRequest.setTotalFee(BaseWxPayRequest.yuanToFen(paymentOrder.getAmount().toString()));
            wxPayUnifiedOrderRequest.setSpbillCreateIp(paymentOrder.getClientIp());
            wxPayUnifiedOrderRequest.setTradeType(WxPayConstants.TradeType.MWEB);

            return wxPayService.createOrder(wxPayUnifiedOrderRequest);
        } catch (WxPayException e) {
            log.error("微信下单失败！订单号：{},原因:{}", paymentOrder.getPaymentNum(), e.getMessage());
            log.error("下单异常：", e);
        }

        return null;
    }

    /**
     * js-api下单
     * @return 订单信息
     */
    WxPayMpOrderResult createJsApiOrder(PaymentOrderDetail paymentOrder){
        try {
            WxPayUnifiedOrderRequest wxPayUnifiedOrderRequest = new WxPayUnifiedOrderRequest();

            wxPayUnifiedOrderRequest.setDeviceInfo("WEB");
            wxPayUnifiedOrderRequest.setOutTradeNo(paymentOrder.getPaymentNum());
            wxPayUnifiedOrderRequest.setBody(wxPayProperties.getBody());
            wxPayUnifiedOrderRequest.setTotalFee(BaseWxPayRequest.yuanToFen(paymentOrder.getAmount().toString()));
            wxPayUnifiedOrderRequest.setSpbillCreateIp(paymentOrder.getClientIp());
            wxPayUnifiedOrderRequest.setTradeType(WxPayConstants.TradeType.JSAPI);

            //支付的公众号授权必须在商户号里面创建联系
            WxUserInfo wxUserInfo = wxMpFacadeService.getWxUserInfoByCode(paymentOrder.getAuthCode(),
                    WechatMpTypeEnum.JS_API.name());
            if (wxUserInfo.getOpenId() == null) {
                log.error("微信授权信息获取失败，code为：[{}]", paymentOrder.getAuthCode());
            }
            wxPayUnifiedOrderRequest.setOpenid(wxUserInfo.getOpenId());

            if(wxPayService.getConfig() != null){
                paymentOrder.setMchId(wxPayService.getConfig().getMchId());
            }

            return wxPayService.createOrder(wxPayUnifiedOrderRequest);

        } catch (WxPayException e) {
            log.error("微信下单失败！订单号：{},原因:{}", paymentOrder.getPaymentNum(), e.getMessage());
            log.error("下单异常：", e);
        }

        return null;
    }
}
