package com.bxm.localnews.payment.service.impl;

import com.bxm.localnews.base.service.BizLogService;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.payment.config.PayProperties;
import com.bxm.localnews.payment.config.WithdrawProperties;
import com.bxm.localnews.payment.constant.WithdrawEnum;
import com.bxm.localnews.payment.constant.WithdrawTypeEnum;
import com.bxm.localnews.payment.domain.WithdrawMapper;
import com.bxm.localnews.payment.dto.WithdrawConfigDTO;
import com.bxm.localnews.payment.dto.WithdrawDTO;
import com.bxm.localnews.payment.dto.WithdrawOptionDTO;
import com.bxm.localnews.payment.param.UserWithdrawParam;
import com.bxm.localnews.payment.param.UserWithdrawV2Param;
import com.bxm.localnews.payment.service.PaymentWithdrawAccountService;
import com.bxm.localnews.payment.service.UserWithdrawService;
import com.bxm.localnews.payment.vo.UserWithdrawAccountInfoVO;
import com.bxm.localnews.payment.vo.WithdrawAccountVO;
import com.bxm.localnews.payment.vo.WithdrawFlow;
import com.bxm.localnews.payment.withdraw.IWithdrawStrategy;
import com.bxm.localnews.payment.withdraw.WithdrawContext;
import com.bxm.localnews.payment.withdraw.WithdrawFilterChain;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSONString;
import static com.bxm.localnews.payment.constant.WithdrawAccountType.ALIPAY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author zhaoyadong 2019/3/11 14:39
 * @desc
 */
@Service
@AllArgsConstructor
public class WithdrawServiceImpl extends BaseService implements UserWithdrawService {

    private final RedisStringAdapter redisStringAdapter;

    private final BizLogService bizLogService;

    private final DistributedLock distributedLock;

    private final PayProperties payProperties;

    private final WithdrawFilterChain withdrawFilterChain;

    private final WithdrawProperties withdrawProperties;

    private final UserAccountIntegrationService userAccountIntegrationService;

    private static final Map<String, IWithdrawStrategy> withdrawStrategyMap = Maps.newHashMap();

    private final IWithdrawStrategy normalWithdrawStrategy;

    private final WithdrawMapper withdrawMapper;

    private final PaymentWithdrawAccountService paymentWithdrawAccountService;

    private final List<IWithdrawStrategy> withdrawStrategyList;

    @PostConstruct
    public void init() {
        withdrawStrategyList.forEach(strategy -> withdrawStrategyMap.put(strategy.match().name(), strategy));
    }

    @Override
    public Message userWithdraw(Long userId, BigDecimal amount, String ip, BasicParam basicParam) {
        UserWithdrawParam param = new UserWithdrawParam();
        param.merge(basicParam);
        param.setUserId(userId);
        param.setRequestIp(ip);

        WithdrawContext context = new WithdrawContext(param);
        context.setAmount(amount);

        return execWithdraw(context);
    }

    @Override
    public Message appletUserWithdraw(Long userId, String openId, BigDecimal amount, String devcId, String ip) {
        UserWithdrawParam param = new UserWithdrawParam();
        param.setUserId(userId);
        param.setRequestIp(ip);
        param.setPlatform(PlatformEnum.APPLET.getCode());
        param.setDevcId(devcId);

        WithdrawContext context = new WithdrawContext(param);
        context.setAmount(amount);
        context.setWithdrawAccount(openId);

        Message message = Message.build();

        //小程序废弃很久了，先不管了
        if (amount.compareTo(new BigDecimal(1)) < 0) {
            return message.setSuccess(false).setMessage("提现金额需要大于或等于1元");
        }

        Double withdrawAmount = redisStringAdapter.increment(getWxMiniAppAmountKey(), amount.doubleValue());
        redisStringAdapter.expire(getWxMiniAppAmountKey(), DateUtils.getCurSeconds());

        if (payProperties.getMiniAppLimitAmount().compareTo(BigDecimal.valueOf(withdrawAmount)) < 0) {
            return message.setSuccess(false).setMessage("今日兑换已达限额，请明日再来");
        }

        message = execWithdraw(context);

        if (Objects.isNull(message) || !message.isSuccess()) {
            redisStringAdapter.decrement(getWxMiniAppAmountKey(), amount.doubleValue());
        }

        return message;
    }

    private Message execWithdraw(WithdrawContext context) {
        // 判断老版本，发起提现的话，提醒用户升级新版本
        Message message = preCheck(context.getOriginParam());
        if (!message.isSuccess()) {
            return message;
        }

        // 3.5.0先强制设置为支付宝提现
        context.setWithdrawType(WithdrawTypeEnum.ALI_WITHDRAW);

        if (logger.isDebugEnabled()) {
            logger.debug("用户发起提现，提现参数为：{}", toJSONString(context));
        }

        String requestId = nextSequence().toString();
        String userIdStr = context.getUserId().toString();

        if (distributedLock.lock(userIdStr, requestId)) {
            //进行提现过滤链处理，判断不同情况下的逻辑处理
            message = withdrawFilterChain.run(context);

            if (null != message && message.isSuccess()) {
                //前置条件验证通过，发起提现,默认使用正常需要审核的方式进行提现
                IWithdrawStrategy withdrawStrategy = withdrawStrategyMap.getOrDefault(context.getWithdrawOption().getStargy(),
                        normalWithdrawStrategy);

                logger.info("用户提现对应的提现策略：{},提现参数：{}",
                        withdrawStrategy.match().name(),
                        toJSONString(context));

                message = withdrawStrategy.execWithdraw(context);

                if (message.isSuccess()) {
                    //用户发起提现的埋点记录
                    bizLogService.initiateWithdrawal(context.getUserId(),
                            context.getAmount(),
                            context.getOriginParam().getPlatform());
                }

            }
            distributedLock.unlock(userIdStr, requestId);
        } else {
            message = Message.build(false, "请稍后再试");
        }

        if (logger.isDebugEnabled()) {
            logger.debug("用户发起提现，提现结果：{},提现参数为：{}",
                    message,
                    toJSONString(context));
        }

        return message;
    }

    @Override
    public Message execWithdraw(UserWithdrawParam param) {
        return execWithdraw(new WithdrawContext(param));
    }

    private Message preCheck(UserWithdrawParam param) {
        // 判断版本是否小于3.5.0
        if (Objects.nonNull(param)
                && isNotBlank(param.getCurVer())
                && StringUtils.isLessThan(param.getCurVer(), "3.5.0")) {

            logger.info("用户: {} 发起了提现申请，但版本过低。提示开关：{},请求参数: {}",
                    param.getUserId(),
                    withdrawProperties.getWithdrawOldVersionNoticeSwitch(),
                    toJSONString(param));

            // 开启了老版本的提醒开关
            if (Boolean.TRUE.equals(withdrawProperties.getWithdrawOldVersionNoticeSwitch())) {
                return Message.build(false).setMessage(withdrawProperties.getWithdrawOldVersionNotice());
            }
        }
        return Message.build(true);
    }

    @Override
    public WithdrawDTO getWithdrawOptions(Long userId) {
        WithdrawDTO withdrawDTO = new WithdrawDTO();
        //设置可提现列表
        setOptions(withdrawDTO, userId);
        //设置用户的当前可用金额
        withdrawDTO.setAvailableCash(userAccountIntegrationService.getUserDrawableCash(userId));
        //设置用户最后一次的提现信息
        WithdrawFlow userLastWithdraw = withdrawMapper.getUserLastWithdraw(userId);
        if (userLastWithdraw != null && WithdrawEnum.FAIL_PAYMENT.getState().equals(userLastWithdraw.getState())) {
            if ("V2_ACCOUNT_SIMPLE_BAN".equals(userLastWithdraw.getRemark())) {
                withdrawDTO.setFailMsg("您的微信号未实名认证，提现失败，请实名认证后再提现");
            } else {
                withdrawDTO.setFailMsg("系统繁忙，请稍后再试");
            }
        }
        // 设置支付宝账号
        Optional<WithdrawAccountVO> paymentAccount = paymentWithdrawAccountService.getPaymentAccount(ALIPAY, userId);
        paymentAccount.ifPresent(withdrawAccountVO -> withdrawDTO.setAlipayAccount(withdrawAccountVO.getAccount()));

        // 提现公告
        if (isNotBlank(withdrawProperties.getWithdrawNote())) {
            withdrawDTO.setNote(withdrawProperties.getWithdrawNote());
            withdrawDTO.setNoteProtocol(withdrawProperties.getWithdrawNoteProtocol());
        }

        return withdrawDTO;
    }

    private void setOptions(WithdrawDTO withdrawDTO, Long userId) {
        //从配置中获取提现列表
        List<WithdrawOptionDTO> sourceOptions = withdrawProperties.getWithdrawOptions();

        sourceOptions = sourceOptions.stream()
                .sorted(Comparator.comparing(WithdrawOptionDTO::getPriority))
                .sorted(Comparator.comparing(WithdrawOptionDTO::getIndex))
                .collect(Collectors.toList());

        List<WithdrawOptionDTO> result = Lists.newArrayList();

        for (int i = 0; i < sourceOptions.size(); i++) {
            for (WithdrawOptionDTO sourceOption : sourceOptions) {
                if (sourceOption.getIndex() == i) {
                    //判断配置的提现方式，用户是否仍可进行提现
                    if (withdrawStrategyMap.get(sourceOption.getStargy()).hasTimes(userId)) {
                        result.add(sourceOption);
                        break;
                    }
                }
            }
        }

        List<WithdrawConfigDTO> configList = result.stream().map(option -> WithdrawConfigDTO.builder()
                .amount(option.getAmount())
                .withdrawId(option.getWithdrawId())
                .enableWithdraw(true)
                .withdrawNumber(1)
                .build())
                .collect(Collectors.toList());

        withdrawDTO.setWithdrawConfigDTOList(configList);
    }

    private KeyGenerator getWxMiniAppAmountKey() {
        return RedisConfig.WEIXIN_MINI_APP_AMOUNT.copy().appendKey("balance");
    }

    @Override
    public Message withdrawV2(UserWithdrawV2Param param) {
        BigDecimal money = param.getMoney();
        if (money == null || money.doubleValue() <= 0) {
            return Message.build(false, "金额必须大于0");
        }
        //舍去后两位
        BigDecimal newMoney = money.setScale(2, BigDecimal.ROUND_DOWN);
        if (newMoney.doubleValue() <= 0) {
            return Message.build(false, "金额必须大于0.01");
        }
        return newVersionUserWithdraw(param);
    }

    private Message newVersionUserWithdraw(UserWithdrawV2Param param) {
        UserWithdrawParam userWithdrawParam = new UserWithdrawParam();
        BeanUtils.copyProperties(param, userWithdrawParam);
        WithdrawContext context = new WithdrawContext(userWithdrawParam);
        context.setAmount(param.getMoney());
        if (param.getType() == 1) {
            context.setWithdrawType(WithdrawTypeEnum.WX_WITHDRAW);
        } else {
            context.setWithdrawType(WithdrawTypeEnum.ALI_WITHDRAW);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("用户发起提现，提现参数为：{}", toJSONString(context));
        }

        String requestId = nextSequence().toString();
        String userIdStr = context.getUserId().toString();

        Message message = null;
        if (distributedLock.lock(userIdStr, requestId)) {
            //进行提现过滤链处理，判断不同情况下的逻辑处理
            message = withdrawFilterChain.run(context);

            if (null != message && message.isSuccess()) {
                //前置条件验证通过，发起提现,默认使用正常需要审核的方式进行提现
                IWithdrawStrategy withdrawStrategy = withdrawStrategyMap.getOrDefault(context.getWithdrawOption().getStargy(),
                        normalWithdrawStrategy);

                logger.info("用户提现对应的提现策略：{},提现参数：{}",
                        withdrawStrategy.match().name(),
                        toJSONString(context));

                message = withdrawStrategy.execWithdraw(context);

                if (message.isSuccess()) {
                    //用户发起提现的埋点记录
                    bizLogService.initiateWithdrawal(context.getUserId(),
                            context.getAmount(),
                            context.getOriginParam().getPlatform());
                }

                distributedLock.unlock(userIdStr, requestId);
            }
        } else {
            message = Message.build(false, "请稍后再试");
        }

        if (logger.isDebugEnabled()) {
            logger.debug("用户发起提现，提现结果：{},提现参数为：{}",
                    message,
                    toJSONString(context));
        }

        return message;
    }

    @Override
    public UserWithdrawAccountInfoVO getWithdrawAccountInfo(Long userId) {
        UserWithdrawAccountInfoVO userWithdrawAccountInfoVO = new UserWithdrawAccountInfoVO();
        Optional<WithdrawAccountVO> paymentAccount = paymentWithdrawAccountService.getPaymentAccount(ALIPAY, userId);
        paymentAccount.ifPresent(withdrawAccountVO -> userWithdrawAccountInfoVO.setAlipayAccount(withdrawAccountVO.getAccount()));
        return userWithdrawAccountInfoVO;
    }
}
