package com.bxm.localnews.payment.withdraw.filter;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.payment.WithdrwaStrategyEnum;
import com.bxm.localnews.payment.config.WithdrawProperties;
import com.bxm.localnews.payment.dto.WithdrawOptionDTO;
import com.bxm.localnews.payment.withdraw.WithdrawContext;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.Objects;

import static com.bxm.newidea.component.tools.StringUtils.isGrateOrEqualThan;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * 最基础的提现过滤器，优先级最高
 *
 * @author liujia
 * @date 2020/05/18 21:59
 */
@Component
@Slf4j
public class BaseWithdrawFilter extends AbstractWithdrawFilter {

    private final UserAccountIntegrationService userAccountIntegrationService;

    private final UserIntegrationService userIntegrationService;

    private final WithdrawProperties withdrawProperties;

    @Autowired
    public BaseWithdrawFilter(UserAccountIntegrationService userAccountIntegrationService,
                              UserIntegrationService userIntegrationService,
                              WithdrawProperties withdrawProperties) {
        this.userAccountIntegrationService = userAccountIntegrationService;
        this.userIntegrationService = userIntegrationService;
        this.withdrawProperties = withdrawProperties;
    }

    @Override
    public Message run(WithdrawContext context) {
        if (!withdrawProperties.getEnableSwitch()) {
            log.info("提现开关未开启---不允许提现");
            return Message.build(false, "请稍后再试");
        }

        //设置用户是否风险用户
        Boolean riskUser = userIntegrationService.isRiskUser(context.getUserId());
        if (riskUser == null) {
            return Message.build(false, "此用户不存在");
        }
        context.setRisk(riskUser);

        //设置提现请求对应的参数
        setOption(context);

        if (context.getAmount().equals(BigDecimal.ZERO)) {
            log.error("用户发起的提现金额不在提现列表中，请求参数：{}", JSON.toJSONString(context));
            return Message.build(false, "请稍后再试");
        }
        //判断余额是否足够
        BigDecimal drawableCash = userAccountIntegrationService.getUserDrawableCash(context.getUserId());
        if (isNotBlank(context.getOriginParam().getCurVer()) && isGrateOrEqualThan(context.getOriginParam().getCurVer(), "3.10.0")) {
            log.info("3.10.0以上版本提现，增加用户达人等级门槛userId:{}", context.getUserId());
            UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(context.getUserId());
            if (userInfoDTO.getTalentLevel() == 0) {
                if (context.getAmount().compareTo(withdrawProperties.getNormalRestrictMoney()) < 0) {
                    return Message.build(false, "满" + withdrawProperties.getNormalRestrictMoney() + "元才可提现，买的越多返的越多哦~");
                }
            } else {
                if (context.getAmount().compareTo(withdrawProperties.getTalentRestrictMoney()) < 0) {
                    return Message.build(false, "满" + withdrawProperties.getNormalRestrictMoney() + "元才可提现，买的越多返的越多哦~");

                }
            }
        }
        if (drawableCash.compareTo(context.getAmount()) < 0) {
            log.error("用户[{}]的账户余额为：[{}]，小于提现金额[{}],不允许提现",
                    context.getUserId(),
                    drawableCash,
                    context.getAmount());
            return Message.build(false, "你的账户余额不足");
        }

        return Message.build();
    }

    /**
     * 设置提现选型，兼容历史逻辑
     *
     * @param context
     */
    private void setOption(WithdrawContext context) {
        for (WithdrawOptionDTO withdrawOption : withdrawProperties.getWithdrawOptions()) {
            if (Objects.equals(context.getOriginParam().getWithdrawId(),
                    withdrawOption.getWithdrawId())) {
                context.setWithdrawOption(withdrawOption);
                return;
            }

            //为了适配老的接口，这里根据金额判断与配置项目是否一致，小程序允许自由填写金额，可能会与某些逻辑冲突
            //考虑到小程序已经狗带，就不管他了
            if (withdrawOption.getAmount().compareTo(context.getAmount()) == 0) {
                context.setWithdrawOption(withdrawOption);
                return;
            }
        }

        //历史逻辑，如果直接设置的金额（非配置金额），小于等于两元并且非风险用户，则走直接提现的逻辑
        if (!context.isRisk()) {
            BigDecimal limitAmount = new BigDecimal("2");
            if (context.getAmount().compareTo(limitAmount) < 1) {
                WithdrawOptionDTO option = new WithdrawOptionDTO();
                option.setAmount(context.getAmount());
                option.setStargy(WithdrwaStrategyEnum.DAILY_ONCE.name());

                context.setWithdrawOption(option);
                return;
            }
        }

        //设置默认的策略
        if (context.getAmount() != null && context.getWithdrawOption() == null) {
            WithdrawOptionDTO option = new WithdrawOptionDTO();
            option.setAmount(context.getAmount());
            option.setStargy(WithdrwaStrategyEnum.NORMAL.name());

            context.setWithdrawOption(option);
        }
    }
}
