package com.bxm.localnews.payment.pay;

import com.bxm.localnews.payment.constant.PayTypeEnum;
import com.bxm.localnews.payment.param.PaymentRefundParam;
import com.bxm.localnews.payment.pay.impl.DefaultPayModeServiceImpl;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.localnews.payment.vo.PaymentOrderDetail;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.Message;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.Map;

/**
 * @author zhaoyadong 2019/1/2 14:01
 * @desc
 */
@Component
@Slf4j
public class PayProxyService {

    private Map<String, PayModeService> payModeMap;

    private PayModeService defaultPayModeService = new DefaultPayModeServiceImpl();

    /**
     * 下单
     * @param paymentOrderDetail
     */
    public void create(PaymentOrderDetail paymentOrderDetail) {
        getService(PayTypeEnum.getNameByType(paymentOrderDetail.getPayType())).create(paymentOrderDetail);
    }

    public void query(PaymentOrder paymentOrder, PayTypeEnum payType) {
        getService(payType.getName()).query(paymentOrder);
    }

    /**
     * 支付完成后的回调处理
     * @param data    回调内容
     * @param payType 支付方式
     * @return 处理结果
     */
    public String notifyPay(String data, PayTypeEnum payType) {
        return getService(payType.getName()).callBack(data);
    }

    /**
     * 获取支付方式对应的支付服务提供接口，如果没有配置服务则返回默认的支付服务接口
     * @param payType 支付方式
     * @return 支付服务提供接口
     */
    private PayModeService getService(String payType) {
        PayModeService service = getPayModeMap().get(payType);

        if (service == null) {
            service = defaultPayModeService;
        }

        return service;
    }



    /**
     * 发起申请退款
     * @param param 支付订单信息
     * @return 回调的处理结果，返回到第三方
     */
    public Message submitRefund(PaymentRefundParam param) {
        Preconditions.checkArgument(null != param);
        Preconditions.checkArgument(null != param.getPayOrder());

        return obtainPayment(PayTypeEnum.getPayTypeEnum(param.getPayOrder().getPayType())).submitRefund(param);
    }

    /**
     * 执行退款申请回调
     * @param request 请求参数
     * @param payType 支付类型
     * @return 回调结果
     */
    public String execRefundCallback(String request, PayTypeEnum payType) {
        return obtainPayment(payType).execRefundCallback(request);
    }

    private PayModeService obtainPayment(PayTypeEnum payType) {
        PayModeService service = getPayModeMap().get(payType.getName());
        if (service == null) {
            log.error("请求的支付方式不存在，支付方式为：{}", payType);
            return defaultPayModeService;
        }
        return service;
    }

    /**
     * 从spring context中加载PayModeService的子类，根据支付类型进行匹配
     * @return 支付类型和对应支付方式服务提供类的map
     */
    private Map<String, PayModeService> getPayModeMap() {
        if (null == payModeMap) {
            payModeMap = Maps.newHashMap();
            Collection<PayModeService> payModeServices = SpringContextHolder.getBeans(PayModeService.class);

            for (PayModeService payModeService : payModeServices) {
                if (payModeService.support() != null) {
                    payModeMap.put(payModeService.support().getName(), payModeService);
                }
            }
        }

        return payModeMap;
    }

}
