package com.bxm.localnews.payment.withdraw.strategy;

import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.util.NidGeneratorUtil;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.param.AccountCashParam;
import com.bxm.localnews.payment.config.WithdrawProperties;
import com.bxm.localnews.payment.constant.WithdrawEnum;
import com.bxm.localnews.payment.constant.WithdrawTypeEnum;
import com.bxm.localnews.payment.domain.WithdrawMapper;
import com.bxm.localnews.payment.pay.WithdrawProxySerivce;
import com.bxm.localnews.payment.vo.WithdrawFlow;
import com.bxm.localnews.payment.withdraw.IWithdrawStrategy;
import com.bxm.localnews.payment.withdraw.WithdrawContext;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;
import java.util.Objects;

import static com.bxm.localnews.payment.constant.WithdrawTypeEnum.ALI_WITHDRAW;

/**
 * @author liujia
 * @date 2020/05/19 09:52
 */
public abstract class AbstractWithdrawStrategy implements IWithdrawStrategy {

    @Autowired
    protected WithdrawMapper withdrawMapper;

    @Autowired
    protected WithdrawProperties withdrawProperties;

    @Autowired
    protected UserAccountIntegrationService userAccountIntegrationService;

    @Autowired
    protected SequenceCreater sequenceCreater;

    @Autowired
    protected WithdrawProxySerivce withdrawProxySerivce;

    /**
     * 直接发起提现汇款
     *
     * @param context 提现上下文
     * @return 提现结果
     */
    protected Message directWithdraw(WithdrawContext context) {
        WithdrawFlow withdrawFlow = buildWithdraw(context);
        withdrawFlow.setState(WithdrawEnum.DURING_PAYMENT.getState());

        withdrawMapper.addWithdrawFlow(withdrawFlow);

        //锁定提现金额
        Message message = userAccountIntegrationService.updateUserWithdraw(context.getUserId(),
                context.getAmount(),
                WithdrawEnum.DURING_PAYMENT.getState());

        if (message.isSuccess()) {
            //直接触发提现
            return withdrawProxySerivce.withdraw(withdrawFlow);
        }

        return message;
    }

    protected void afterThirdpartyPay(WithdrawContext context, WithdrawFlow withdrawFlow, boolean success, String reason) {
        // 修改状态为成功 or 失败
        WithdrawFlow update = new WithdrawFlow();
        update.setId(withdrawFlow.getId());
        if (success) {
            // 提现成功
            update.setState(WithdrawEnum.SUCCESS_PAYMENT.getState());
            userAccountIntegrationService.updateUserWithdraw(context.getUserId(),
                    context.getAmount(),
                    WithdrawEnum.SUCCESS_PAYMENT.getState());
        } else {
            // 提现失败
            update.setState(WithdrawEnum.FAIL_PAYMENT.getState());
            update.setRemark(reason);
            userAccountIntegrationService.updateUserWithdraw(context.getUserId(),
                    context.getAmount(),
                    WithdrawEnum.FAIL_PAYMENT.getState());
        }
        withdrawMapper.updateWithdrawFlow(update);
    }

    /**
     * 将提现申请提交审核
     *
     * @param context 提现上下文
     * @return 提现申请结果
     */
    protected Message submitApprove(WithdrawContext context) {
        WithdrawFlow withdrawFlow = buildWithdraw(context);
        //增加提现订单
        int result = withdrawMapper.addWithdrawFlow(withdrawFlow);
        //更新账户信息
        updateUserAccount(context);

        return Message.build(result);
    }

    /**
     * 提现更新账户信息，将可提现金额变更为提现中的金额（审核状态）
     *
     * @param context 提现上下文信息
     */
    protected void updateUserAccount(WithdrawContext context) {
        AccountCashParam accountCashParam = new AccountCashParam(context.getUserId(),
                "DRAWABLEL_CASH",
                Boolean.FALSE,
                context.getAmount(),
                context.getWithDrawId(),
                Objects.equals(ALI_WITHDRAW, context.getWithdrawType()) ? "ALIPAY_WITHDRAWAL" : "WECHAT_WITHDRAWAL",
                null);
        userAccountIntegrationService.addCash(accountCashParam);
    }

    protected WithdrawFlow buildWithdraw(WithdrawContext context) {
        WithdrawFlow withdrawFlow = new WithdrawFlow();
        withdrawFlow.setId(sequenceCreater.nextLongId());
        withdrawFlow.setOrderNo(generateWithdrawNum());
        // 如不指定类型 默认使用微信支付
        withdrawFlow.setPayChannel(Objects.isNull(context.getWithdrawType()) ? WithdrawTypeEnum.WX_WITHDRAW.getId()
                : context.getWithdrawType().getId());
        withdrawFlow.setUserId(context.getUserId());
        withdrawFlow.setPayAccount(context.getWithdrawAccount());
        withdrawFlow.setState(WithdrawEnum.AUDIT.getState());
        //风险用户直接进入延迟审核
        if (context.isRisk()) {
            withdrawFlow.setState(WithdrawEnum.DELAY_AUDIT.getState());
        }
        withdrawFlow.setAmount(context.getAmount());
        withdrawFlow.setRealName(context.getRealName());
        withdrawFlow.setClientIp(context.getOriginParam().getRequestIp());
        withdrawFlow.setEquipment(context.getOriginParam().getDevcId());
        Date now = new Date();
        withdrawFlow.setCreateTime(now);
        withdrawFlow.setUpdateTime(now);
        withdrawFlow.setOpenIdType(context.getOpenIdType());
        //后续逻辑有使用到提现流水ID
        context.setWithDrawId(withdrawFlow.getId());

        return withdrawFlow;
    }

    /**
     * 生成订单号
     *
     * @return
     */
    protected String generateWithdrawNum() {
        String prefix = WithdrawTypeEnum.getNameByType(WithdrawTypeEnum.WX_WITHDRAW.getType());
        return NidGeneratorUtil.getOrderNo(prefix);
    }
}
