package com.bxm.localnews.quartz.service.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.bxm.component.mybatis.utils.BatchHelper;
import com.bxm.localnews.quartz.config.NewsProperties;
import com.bxm.localnews.quartz.domain.NewsPoolMapper;
import com.bxm.localnews.quartz.service.NewsPoolservice;
import com.bxm.localnews.quartz.vo.NewsPool;
import com.bxm.newidea.component.service.BaseService;
import com.google.common.collect.Lists;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

/**
 * 新闻缓冲池实现
 */
@Service
@RefreshScope
public class NewsPoolServiceImpl extends BaseService implements NewsPoolservice {

    /**
     * 缓存池中推荐类别最大数量
     */
    private static final int MAX_POOL_SIZE = 5000;

    /**
     * 缓存池中具体类别最大数量
     */
    private static final int MAX_KIND_POOL_SIZE = 3000;

    @Autowired
    private NewsPoolMapper newsPoolMapper;

    @Autowired
    private NewsProperties newsProperties;

    @Override
    public void fillNewsPool() {
        List<Integer> kindIds = newsPoolMapper.listKinds();
        ExecutorService fixedThreadPool = Executors.newFixedThreadPool(20);
        for (Integer kindId : kindIds) {
            fixedThreadPool.execute(() -> {
                //每小时每次取1000条记录
                Map<String, Object> param = new HashMap<>();
                param.put("kindId", kindId);
                param.put("isHot", 1);
                param.put("limit", 1000);
                List<NewsPool> newsList = newsPoolMapper.recommendNews(param);

                int count = newsPoolMapper.getCount(param);
                if (count > 0) {
                    int diff = newsList.size() + count - MAX_KIND_POOL_SIZE;
                    if (diff > 0) {
                        //删除旧的，保持池中的数量
                        param.put("limit", diff);
                        newsPoolMapper.deleteExcess(param);
                    }
                }

                if (kindId.equals(newsProperties.getRecommendKindId())) {
                    newsList.addAll(fillRecommedNewsPool());
                    Collections.shuffle(newsList);
                }
                new BatchHelper<NewsPoolMapper, NewsPool>(NewsPoolMapper.class, newsList) {
                    @Override
                    protected int invoke(NewsPool element) {
                        return this.mapper.insert(element);
                    }
                };
            });
        }
        fixedThreadPool.shutdown();
        try {
            while (!fixedThreadPool.isTerminated()) {
                Thread.sleep(150);
            }
        } catch (Exception e) {
            logger.error("an error occured in fixedThreadPool: {}", e);
        }
    }

    private List<NewsPool> fillRecommedNewsPool() {
        Map<String, Object> param = new HashMap<>();
        param.put("isHot", 2);
        param.put("limit", 1000);
        List<NewsPool> newsList = newsPoolMapper.recommendNews(param);
        List<NewsPool> newsPoolList = Lists.newArrayList();
        int count = newsPoolMapper.getCount(param);
        if (count > 0) {
            int diff = count + newsList.size() - MAX_POOL_SIZE;
            if (diff > 0) {
                //删除旧的，保持池中的数量
                param.put("limit", diff);
                newsPoolMapper.deleteExcess(param);
            }
        }

        return newsPoolList;
    }

}
