package com.bxm.localnews.quartz.service.impl;

import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.quartz.domain.RedPacketMapper;
import com.bxm.localnews.quartz.facade.PushMsgIntegService;
import com.bxm.localnews.quartz.service.RedPacketQuartzService;
import com.bxm.localnews.quartz.vo.RedPacket;
import com.bxm.newidea.component.uuid.SequenceCreater;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

/**
 * 红包定时任务服务实现类
 * @author donghy
 */
@Service
public class RedPacketQuartzServiceImpl implements RedPacketQuartzService {

    /**
     * 日志对象
     */
    private static final Logger LOG = LoggerFactory.getLogger(RedPacketQuartzServiceImpl.class);

    /**
     * 红包数据接口层
     */
    private RedPacketMapper redPacketMapper;

    /**
     * 序列号创建服务
     */
    private SequenceCreater sequenceCreater;

    /**
     * 消息推送服务
     */
    private PushMsgIntegService pushMsgIntegService;

    @Autowired
    RedPacketQuartzServiceImpl(RedPacketMapper redPacketMapper, SequenceCreater sequenceCreater,
                               PushMsgIntegService pushMsgIntegService) {
        this.redPacketMapper = redPacketMapper;
        this.sequenceCreater = sequenceCreater;
        this.pushMsgIntegService = pushMsgIntegService;
    }

    @Override
    public void pushMessage() {

        LOG.info("红包推送服务定时任务执行开始");

        //查询符合条件的红包List
        List<RedPacket> pushRedPacketList = redPacketMapper.queryPushList();
        LOG.info("红包推送服务定时任务执行,待推送的红包数量为:{}", pushRedPacketList.size());

        //若待执行的红包数量为空,则直接返回
        if (CollectionUtils.isEmpty(pushRedPacketList)) {
            LOG.info("待推送的红包列表为空");
            return;
        }

        //遍历List
        pushRedPacketList.forEach(redPacket -> {
            //===发送推送消息
            pushMessageToClient(redPacket);
            //===更新push标志位1
            redPacketMapper.updatePushFlag(redPacket.getId());
        });

    }

    /**
     * 推送消息到客户端
     * @param redPacket 红包信息
     */
    private void pushMessageToClient(RedPacket redPacket){

        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.RECEIVE_RED_PACKET);
        String content = String.format("\uD83D\uDE09哇哦，你的红包已经被%s人领取了，看看都有谁", redPacket.getActualNum());

        Long msgId = sequenceCreater.nextLongId();

        //=======推送消息
        info.setMsgId(msgId);
        info.addExtend("msgId", msgId);
        info.addExtend("rpId", redPacket.getId());
        info.addExtend("userId", redPacket.getUserId());

        PushMessage pushMessage = PushMessage.build();
        pushMessage.setTitle("");
        pushMessage.setContent(content);
        pushMessage.setType(TemplateTypeEnum.NOTIFCTION);
        pushMessage.setPushReceiveScope(PushReceiveScope.pushSignle(redPacket.getUserId()));
        pushMessage.setPayloadInfo(info);

        pushMsgIntegService.pushMsg(pushMessage);
    }

}
