package com.bxm.newidea.aspect;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;

/**
 * aop采集日志(接口请求输入,输出，接口调用时间)
 */
@Aspect
@Component
public class HttpAspect {

    private static final Logger logger = LoggerFactory.getLogger(HttpAspect.class);

    private static ThreadLocal<Long> startTime = new ThreadLocal<>();

    @Pointcut("execution(public * com.bxm.newidea.controller.*Controller.*(..))")
    public void log() {
    }

    /**
     * @param joinPoint 注意：该方法中的HttpServletRequest为javax.servlet.http.HttpServletRequest;
     */
    @Before("log()")
    @SuppressWarnings("{}")
    public void doBefore(JoinPoint joinPoint) {
        startTime.set(System.currentTimeMillis());
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        if (attributes != null) {
            HttpServletRequest request = attributes.getRequest();
            //记录请求路径url
            StringBuffer url = request.getRequestURL();
            //logger.debug("url={}", request.getRequestURL());
            //记录请求方式method
            String method = request.getMethod();
            //logger.debug("method={}", request.getMethod());
            //记录访问者ip
            String remoteAddr = request.getRemoteAddr();
            //logger.debug("ip={}", request.getRemoteAddr());
            //记录访问的类方法
            String classMethod = joinPoint.getSignature().getDeclaringTypeName() + "." + joinPoint.getSignature().getName();
            //logger.debug("class_method={}", joinPoint.getSignature().getDeclaringTypeName() + "." + joinPoint.getSignature().getName());
            //记录传递的参数
            Object[] args = joinPoint.getArgs();
            logger.debug("url={},method={},ip={},class_method={},args={}", url, method, remoteAddr, classMethod, args);
        }

    }

    @After("log()")
    public void doAfter() {

    }

    @AfterReturning(returning = "obj", pointcut = "log()")
    public void doAfterReturning(Object obj) {
        logger.debug("response={}", obj);
        logger.info("响应时长={}ms", System.currentTimeMillis() - startTime.get());
        startTime.remove();
    }


}
