package com.bxm.newidea.recommend;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redis.impl.DefaultKeyGenerator;
import com.bxm.newidea.constant.RedisKeyConstant;
import com.bxm.newidea.domain.NewsBlackMapper;
import com.bxm.newidea.domain.VideoBlackMapper;
import com.bxm.newidea.dto.VideoDto;
import com.bxm.newidea.vo.NewsBlack;
import com.google.common.hash.BloomFilter;
import com.google.common.hash.Funnels;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 数据过滤器 信息召回 去重
 */
@Component
public class RecommendFilter {
    private RedisListAdapter redisListAdapter;

    private BloomFilter<Long> globalNewsBlackFilter;

    private BloomFilter<Long> globalVideoBlackFilter;

    private NewsBlackMapper newsBlackMapper;

    private VideoBlackMapper videoBlackMapper;

    private KeyGenerator newsKeyGenerator = RedisKeyConstant.NEWS_BLACK.copy();

    private KeyGenerator videoKeyGenerator =  RedisKeyConstant.VIDEO_BLACK.copy();

    @Autowired(required = false)
    public RecommendFilter(RedisListAdapter redisListAdapter, NewsBlackMapper newsBlackMapper, VideoBlackMapper videoBlackMapper) {
        this.redisListAdapter = redisListAdapter;
        this.newsBlackMapper = newsBlackMapper;
        this.videoBlackMapper = videoBlackMapper;
    }

    /**
     * 使用全局的新闻过滤器过滤掉推荐的新闻结果
     * @param ids 新闻id集合
     * @return 过滤后的新闻id集合
     */
    public List<Long> newsFilter(List<Long> ids){
        long size = redisListAdapter.size(newsKeyGenerator);
        if(size==0||globalNewsBlackFilter==null){
            refreshNewsBlackCache();
       }
      return  ids.stream().filter(id->!globalNewsBlackFilter.mightContain(id)).collect(Collectors.toList());
    }

    /**
     * 缺省一小时刷新一次全局新闻黑名单的过滤器 也可以调用此方法主动刷新
     */
    public void refreshNewsBlackCache(){
        List<Long> ids = newsBlackMapper.queryNewsBlackIdList();
      //  List<Long> longs = ids.stream().map(NewsBlack::getNewsId).boxed().collect(Collectors.toList());
        redisListAdapter.rightPush(newsKeyGenerator,ids.toArray(new Long[0]));
        redisListAdapter.expire(newsKeyGenerator,3600);
        globalNewsBlackFilter = BloomFilter.create(Funnels.longFunnel(), ids.size(),0.0001);
        ids.forEach(id->globalNewsBlackFilter.put(id));
    }
    /**
     * 使用全局的新闻过滤器过滤掉推荐的小视频结果
     * @param ids 视频dto集合
     * @return 过滤后的视频dto集合
     */
    public List<VideoDto> videoFilter(List<VideoDto> ids){
        long size = redisListAdapter.size(videoKeyGenerator);
        if(size==0||globalVideoBlackFilter==null){
            refreshVideoBlackCache();
        }
        return  ids.stream().filter(id->!globalVideoBlackFilter.mightContain(id.getId())).collect(Collectors.toList());
    }
    /**
     * 缺省一小时刷新一次全局小视频黑名单的过滤器 也可以调用此方法主动刷新
     */
    public void refreshVideoBlackCache(){
        List<Long> ids = videoBlackMapper.queryVideoBlackIdList();
        redisListAdapter.rightPush(videoKeyGenerator,ids.toArray(new Long[0]));
        redisListAdapter.expire(videoKeyGenerator,3600);
        globalVideoBlackFilter = BloomFilter.create(Funnels.longFunnel(), ids.size(),0.0001);
        ids.forEach(id->globalVideoBlackFilter.put(id));
    }
}
