package com.bxm.newidea.recommend;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.constant.RedisKeyConstant;
import com.bxm.newidea.domain.NewsRecommendedMapper;
import com.bxm.newidea.dto.VideoDto;
import com.bxm.newidea.recommend.handler.news.TopNewsRecommender;
import com.bxm.newidea.vo.News;
import com.google.common.hash.BloomFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

public abstract class AbstractNewsRecommender extends AbstractRecommender {

    protected boolean recommended = true;

    protected Logger logger = LoggerFactory.getLogger(this.getClass());

    protected RedisListAdapter redisListAdapter;

    protected NewsRecommendedMapper newsRecommendedMapper;

    public AbstractNewsRecommender(double weight, int sort) {
        super(weight, sort);
    }

    @Override
    public List<Long> recommendNews(Long userId, Integer num) {
        try {
            long begin = System.currentTimeMillis();
            //同步推荐获取结果集
            List<Long> newsIds = syncRecommendNews(userId, null, num, null);
            //本次推荐结果写入已推荐缓存
            if (newsIds != null && !newsIds.isEmpty()&&recommended) {
                KeyGenerator recommendedNewsKey = RedisKeyConstant.NEWS_RECOMMENDED.copy().appendKey(userId);
                redisListAdapter.rightPush(recommendedNewsKey, newsIds.toArray(new Long[0]));
                newsRecommendedMapper.batchInsert(userId, newsIds);
            }
            AbstractNewsRecommender recommender = SpringContextHolder.getBean(this.getClass());
            //异步推荐
            recommender.asyncRecommendNews(userId, null, null);
            logger.info("推荐结果：{} 耗时：{}", newsIds == null ? 0 : newsIds.size(), System.currentTimeMillis() - begin);
            return newsIds;
        }catch (Exception e){
            logger.error("推荐策略发生异常：{}",e);
        }
        return new ArrayList<>();
    }

    @Override
    public List<Long> recommendNews(Long userId, Integer kindId, Integer num,String areaCode) {
        long begin = System.currentTimeMillis();
        //同步推荐获取结果集
        List<Long> newsIds = syncRecommendNews(userId,kindId,num,areaCode);
        //本次推荐结果写入已推荐缓存
        if(newsIds!=null&&!newsIds.isEmpty()) {
            KeyGenerator recommendedNewsKey = RedisKeyConstant.NEWS_RECOMMENDED.copy().appendKey(userId);
            redisListAdapter.rightPush(recommendedNewsKey, newsIds.toArray(new Long[0]));
            newsRecommendedMapper.batchInsert(userId, newsIds);
        }
        //异步推荐
        AbstractNewsRecommender recommender = SpringContextHolder.getBean(this.getClass());
        recommender.asyncRecommendNews(userId,kindId,areaCode);
        logger.info("推荐结果：{} 耗时：{}",newsIds==null?0:newsIds.size(),System.currentTimeMillis()-begin);
        return newsIds;
    }

    /**
     * 同步推荐
     * @param userId 用户id
     * @param kindId 频道id
     * @param num 推荐数量
     * @return 推荐结果(newsId list)
     */
    protected abstract List<Long> syncRecommendNews(Long userId,Integer kindId, Integer num,String areaCode);

    /**
     * 异步推荐
     * @param userId 用户id
     * @param kindId 频道id
     */
    protected abstract void asyncRecommendNews(Long userId,Integer kindId,String areaCode);

    @Override
    public List<VideoDto> recommendVideo(Long user, Integer num) {

        return null;
    }

    /**
     * 布隆过滤器去重
     * @param keyGenerator 拉取的redis key名
     * @param num 数量
     * @param bloomFilter 已填充好的bloomfilter
     * @param resultList 过滤清洗的结果集合
     */
    protected  void recommendedFilter(KeyGenerator keyGenerator, Integer num, BloomFilter<Long> bloomFilter, List<Long> resultList){
        long begin = System.currentTimeMillis();
        List<News> newsList = redisListAdapter.leftIndex(keyGenerator, num - 1, News.class);
        logger.info("redis拉取列表耗时：{} 列表长：{}",System.currentTimeMillis()-begin,newsList.size());
        if(newsList.isEmpty()){
            return;
        }
        int flag = 0;

        for(News news:newsList) {
            if(bloomFilter!=null&&bloomFilter.mightContain(news.getId())){
                //已经推荐过
                flag++;
                continue;
            }
            resultList.add(news.getId());
        }
        long recommendSize = redisListAdapter.size(keyGenerator);
        redisListAdapter.leftTrim(keyGenerator, num, recommendSize);
        if(flag>0){
            recommendedFilter(keyGenerator,flag,bloomFilter,resultList);
        }
        logger.info("bloom去重耗时：{}",System.currentTimeMillis()-begin);
    }
}
