package com.bxm.newidea.recommend.engine;

import com.alibaba.fastjson.JSONObject;
import com.bxm.newidea.enums.NewsKindEnum;
import com.bxm.newidea.integration.NewsSyncIntegrationService;
import com.bxm.newidea.param.NewsRecommendParam;
import com.bxm.newidea.recommend.AbstractRecommender;
import com.bxm.newidea.recommend.filter.RecommendFilter;
import com.bxm.newidea.recommend.RecommendManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.stream.Collectors;

@Component
public class NewsRecommendEngine {

    private Logger logger = LoggerFactory.getLogger(NewsRecommendEngine.class);

    private RecommendFilter recommendFilter;

    private RecommendManager recommendManager;

    private NewsSyncIntegrationService newsSyncIntegrationService;


    @Autowired
    public NewsRecommendEngine(RecommendManager recommendManager,
                               RecommendFilter recommendFilter,
                               NewsSyncIntegrationService newsSyncIntegrationService) {
        this.recommendManager = recommendManager;
        this.recommendFilter = recommendFilter;
        this.newsSyncIntegrationService = newsSyncIntegrationService;
    }

    /**
     * 入口方法
     * 判断传过来的kindId是多少
     * 如果是36或者null则代表是首页推荐，如果是具体数值则代表是各频道内的推荐
     * @param userId
     * @param kindId
     * @param num
     * @param areaCode
     * @param curPage
     * @return
     */
    public List<Long> recommendNews(Long userId, Integer kindId, Integer num, String areaCode, Integer curPage) {
        NewsRecommendParam newsRecommendParam = new NewsRecommendParam().setUserId(userId).setNum(num).setAreaCode(areaCode).setCurPage(curPage).setKindId(kindId);
        // 1.如果是本地标签则只执行本地新闻推荐(暂时取消)
        if (kindId != null && kindId.intValue() == NewsKindEnum.LOCAL.getCode().intValue()) {
            return this.recommendLocalNews(newsRecommendParam);
        } else {
            // 2.首页推荐频道将kindId置空，兼容以前逻辑
            if (kindId == null || (kindId.equals(NewsKindEnum.RECOMMEND.getCode()))) {
                newsRecommendParam.setKindId(null);
                return this.recommendOldHomeNews(newsRecommendParam);
            }
            //3.各频道的推荐
            return this.recommendKindNews(newsRecommendParam);
        }
    }

    /**
     * 频道推荐
     * @return
     */
    private List<Long> recommendKindNews(NewsRecommendParam param) {
        int num = param.getNum();
        Long userId = param.getUserId();
        //1.获取频道推荐的推荐器列表
        List<AbstractRecommender> recommenderWorkers = recommendManager.getNewsRecommenders();

        //2.得到新闻推荐出来的新闻id列表
        List<Long> newsIds = getNewsList(recommenderWorkers,param);

        //5.全局黑名单过滤
        newsIds = recommendFilter.newsFilter(newsIds);

        //6.去重
        newsIds = removeDuplicateNews(newsIds);

        //7.存入数据库
        this.newsSyncIntegrationService.generateRecommneded(userId, newsIds);

        return newsIds.size() > num ? newsIds.subList(0, num) : newsIds;
    }

    /**
     * 首页推荐（旧版-首页中的频道）
     * @return
     */
    private List<Long> recommendHomeKindNews(NewsRecommendParam param) {
        int num = param.getNum();
        //1.获取首页推荐的推荐器列表
        List<AbstractRecommender> recommenderWorkers = recommendManager.getCommonNewsRecommenders();

        //2.得到新闻推荐出来的新闻id列表
        List<Long> newsIds = getNewsList(recommenderWorkers, param);

        //3.截取
        return newsIds.size() > num ? newsIds.subList(0, num) : newsIds;
    }

    /**
     * 首页推荐（旧版）
     * @return
     */
    private List<Long> recommendOldHomeNews(NewsRecommendParam param) {
        int num = param.getNum();
        Long userId = param.getUserId();
        String areaCode = param.getAreaCode();
        Integer curPage = param.getCurPage();
        Integer kindId = param.getKindId();

        List<Long> commonNewsIds = new LinkedList<>();

        //1.取得本地新闻
        NewsRecommendParam localNewsRecommendParam = new NewsRecommendParam().setUserId(userId).setKindId(kindId).setNum(3).setAreaCode(areaCode).setCurPage(curPage);
        List<Long> localNewsIds = recommendLocalNews(localNewsRecommendParam);

        //2.取得热门新闻（如果是第一页的话）
        NewsRecommendParam hotNewsRecommendParam = new NewsRecommendParam().setUserId(userId).setKindId(kindId).setNum(2).setAreaCode(areaCode).setCurPage(curPage);
        List<Long> hotNewsIdList = this.getHotNewsList(hotNewsRecommendParam);

        //3.多线程获取各频道新闻
        commonNewsIds.addAll(getEachKindNewsList(userId, areaCode, curPage));

        //4.把本地新闻穿插进去
        this.interspersedLocalNews(localNewsIds, commonNewsIds);

        //5.前置增加热门新闻
        if (!CollectionUtils.isEmpty(hotNewsIdList)) {
            ((LinkedList<Long>) commonNewsIds).addFirst(hotNewsIdList.get(0));
        }

        //6.填充剩余条数(这里增加10条是防止有重复的)
        int fetchNum = num - commonNewsIds.size() + 10;
        NewsRecommendParam otherNewsRecommendParam = new NewsRecommendParam().setUserId(userId).setNum(fetchNum).setAreaCode(areaCode);
        commonNewsIds.addAll(recommendHomeKindNews(otherNewsRecommendParam));

        //7.全局黑名单过滤
        commonNewsIds = recommendFilter.newsFilter(commonNewsIds);

        //8.去重
        commonNewsIds = commonNewsIds.stream().distinct().limit(num).collect(Collectors.toList());

        //9.存入数据库
        this.newsSyncIntegrationService.generateRecommneded(userId, commonNewsIds);

        return commonNewsIds;
    }


    /**
     * 遍历推荐器列表循环获取新闻
     *
     * @param recommenderWorkers
     * @return
     */
    private List<Long> getNewsList(List<AbstractRecommender> recommenderWorkers,NewsRecommendParam newsRecommendParam) {
        List<Long> newsIds = new LinkedList<>();

        Iterator<AbstractRecommender> iterator = recommenderWorkers.iterator();
        int recommendNum = 0;
        while (iterator.hasNext()) {
            AbstractRecommender recommender = iterator.next();

            List<Long> results = getRecommendNewsList(recommender,newsRecommendParam, recommendNum);

            recommendNum += results.size();
            newsIds.addAll(results);

            if (newsIds.size() >= newsRecommendParam.getNum()) {
                break;
            }
        }
        return newsIds;
    }

    /**
     * 根据某个推荐器获取新闻id列表
     *
     * @param recommender
     * @param recommendNum 已推荐数量
     * @return
     */
    private List<Long> getRecommendNewsList(AbstractRecommender recommender,
                                            NewsRecommendParam newsRecommendParam,
                                            int recommendNum) {
        int num = newsRecommendParam.getNum();
        //1.获取推荐器权重对应的推荐数量
        int fetchNum = new BigDecimal(num).multiply(new BigDecimal("" + recommender.getWeight())).intValue();

        //2.如果权重推荐数量大于剩余推荐数量，则选择剩余数量
        if (num - recommendNum < fetchNum) {
            fetchNum = num - recommendNum;
        }

        //3.获取新闻id
        logger.debug("{} 预计推荐数量：{} ", recommender.getClass(), JSONObject.toJSON(newsRecommendParam));
        newsRecommendParam.setNum(fetchNum+2);
        List<Long> results = recommender.recommend(newsRecommendParam);
        logger.debug("{} 实际推荐数量：{} 用户：{}", recommender.getClass(), results.size(), newsRecommendParam.getUserId());

        return results;
    }

    /**
     * 获得各个频道的新闻id
     *
     * @param userId
     * @param areaCode
     * @param curPage
     * @return
     */
    private List<Long> getEachKindNewsList(Long userId, String areaCode, int curPage) {
        List<Integer> kindList = this.getKindList(curPage);

        List<List<Long>> newsKindList = getStageKindList(kindList).thenCompose(cars -> {
            List<CompletionStage<List<Long>>> kindCompletionStages = kindList.stream()
                    .map(kindId -> getNewsId(userId, kindId, 1, areaCode)).collect(Collectors.toList());

            CompletableFuture<Void> done = CompletableFuture
                    .allOf(kindCompletionStages.toArray(new CompletableFuture[0]));
            return done.thenApply(v -> kindCompletionStages.stream().map(CompletionStage::toCompletableFuture)
                    .map(CompletableFuture::join).collect(Collectors.toList()));
        }).whenComplete((lists, th) -> {
            if (th == null) {
                logger.debug("成功通过多线程获取多个频道的新闻:{}", JSONObject.toJSON(lists));
            }
//            else {
//                throw new RuntimeException(th);
//            }
        }).toCompletableFuture().join();

        List<Long> newsList = new ArrayList<>();
        for (List<Long> e : newsKindList) {
            newsList.addAll(e);
        }
        return newsList;
    }

    /**
     * 得到新闻id
     *
     * @param userId
     * @param kindId
     * @param num
     * @param areaCode
     * @return
     */
    CompletionStage<List<Long>> getNewsId(Long userId, int kindId, int num, String areaCode) {
        NewsRecommendParam newsRecommendParam = new NewsRecommendParam().setAreaCode(areaCode).setNum(num).setKindId(kindId).setUserId(userId);
        return CompletableFuture.supplyAsync(() -> recommendHomeKindNews(newsRecommendParam));
    }

    /**
     * 得到新闻频道id
     *
     * @param kindList
     * @return
     */
    private CompletionStage<List<Integer>> getStageKindList(List<Integer> kindList) {
        return CompletableFuture.supplyAsync(() -> kindList);
    }

    /**
     * 得到热门新闻
     * @return
     */
    private List<Long> getHotNewsList(NewsRecommendParam newsRecommendParam) {
        if (1 == newsRecommendParam.getCurPage()) {
            return this.recommendHotNews(newsRecommendParam);
        }
        return null;
    }

    /**
     * 得到频道id列表
     *
     * @param curPage
     * @return
     */
    private List<Integer> getKindList(int curPage) {
        List<Integer> kindList;
        if (1 == curPage) {
            kindList = generateHomeKind();
        } else {
            kindList = generateKind();
        }
        return kindList;
    }

    /**
     * 穿插本地新闻
     *
     * @param localNewsIds
     * @param commonNewsIds
     */
    private void interspersedLocalNews(List<Long> localNewsIds, List<Long> commonNewsIds) {
        if (!CollectionUtils.isEmpty(localNewsIds)) {
            int count = localNewsIds.size();
            int index = 0;
            for (int i = 0; i < count; i++) {
                if (index > commonNewsIds.size() - 1) {
                    ((LinkedList<Long>) commonNewsIds).addLast(localNewsIds.get(i));
                } else {
                    commonNewsIds.add(index, localNewsIds.get(i));
                    index = index + 2;
                }
            }
        }
    }

    /**
     * 得到第二页的新闻频道(可以放在配置文件中)
     *
     * @return
     */
    private List<Integer> generateKind() {
        List<Integer> kindList = new ArrayList<>();
        kindList.add(32);
        kindList.add(20);
        kindList.add(18);
        kindList.add(14);
        kindList.add(69);
        kindList.add(27);
        return kindList;
    }

    /**
     * 得到第一页的新闻频道(可以放在配置文件中)
     *
     * @return
     */
    private List<Integer> generateHomeKind() {
        List<Integer> kindList = new ArrayList<>();
        kindList.add(69);
        kindList.add(18);
        kindList.add(69);
        kindList.add(27);
        return kindList;
    }

    /**
     * @return
     * @Description 方法描述：推荐本地新闻
     * @author leon 2019年1月23日 下午3:06:53
     * @CopyRight 杭州微财网络科技有限公司
     */

    private List<Long> recommendLocalNews(NewsRecommendParam newsRecommendParam) {
        AbstractRecommender recommender = recommendManager.getLocalNewsRecommender();
        List<Long> results = recommender.recommend(newsRecommendParam);
        return results;
    }

    /**
     * 热门新闻推荐
     * @return
     */
    private List<Long> recommendHotNews(NewsRecommendParam newsRecommendParam) {
        AbstractRecommender recommender = recommendManager.getHotNewsRecommender();
        List<Long> results = recommender.recommend(newsRecommendParam);
        return results;
    }

    /**
     * 去重
     * @param list
     * @return
     */
    private List<Long> removeDuplicateNews(List<Long> list) {
        return list.stream().distinct().collect(Collectors.toList());
    }




}
