package com.bxm.newidea.recommend.framework;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.config.NewsCacheThresholdConfig;
import com.bxm.newidea.constant.RedisKeyConstant;
import com.bxm.newidea.integration.NewsSyncIntegrationService;
import com.bxm.newidea.param.NewsRecommendParam;
import com.bxm.newidea.recommend.AbstractRecommender;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

public abstract class AbstractNewsRecommender extends AbstractRecommender {

    protected Logger logger = LoggerFactory.getLogger(this.getClass());

    protected RedisListAdapter redisListAdapter;

    protected NewsSyncIntegrationService newsSyncIntegrationService;

    @Autowired
    protected RedisSetAdapter redisSetAdapter;

    public AbstractNewsRecommender(double weight, int sort) {
        super(weight, sort);
    }

    @Override
    public List<Long> recommend(NewsRecommendParam param) {
        long begin = System.currentTimeMillis();

        int num = param.getNum();
        Long userId = param.getUserId();
        String areaCode = param.getAreaCode();
        Integer kindId = param.getKindId();

        // 1.同步推荐获取结果集
        List<Long> ids = syncRecommend(userId, kindId, num, areaCode);
        // 2.存入redis推荐记录表
        this.asyncProcessing(userId, ids);
        // 3.异步推荐(仅针对redis缓存池)
        AbstractNewsRecommender recommender = SpringContextHolder.getBean(this.getClass());
        recommender.asyncRecommend(userId, kindId, areaCode);


        logger.info("{}推荐器推荐结果:{}, 参数:{}, 耗时:{}",
                this.getClass().getSimpleName(),
                ids == null ? 0 : JSONObject.toJSONString(ids),
                JSONObject.toJSON(param),
                System.currentTimeMillis() - begin);
        return ids;
    }

    /**
     * 同步推荐
     *
     * @param userId 用户id
     * @param kindId 频道id
     * @param num    推荐数量
     * @param areaCode
     * @return 推荐结果(newsId list)
     */
    protected abstract List<Long> syncRecommend(Long userId, Integer kindId, Integer num, String areaCode);

    /**
     * 异步推荐
     *
     * @param userId 用户id
     * @param kindId 频道id
     * @param areaCode
     */
    protected abstract void asyncRecommend(Long userId, Integer kindId, String areaCode);

    /**
     * 存入redis推荐记录表
     * @param userId
     * @param ids
     */
    private void asyncProcessing(Long userId, List<Long> ids) {
        if (ids != null && !ids.isEmpty()) {
            KeyGenerator recommendedNewsKey = RedisKeyConstant.NEWS_NEW_RECOMMENDED.copy().appendKey(userId);
            redisSetAdapter.add(recommendedNewsKey, ids.toArray(new Long[0]));
            redisSetAdapter.expire(recommendedNewsKey, NewsCacheThresholdConfig.NEWS_EXPIRE_RECOMMENDED_RECORD);
        }
    }

    /**
     * 对redis中的缓存数据进行去重处理
     * @param newsPoolKey redis缓存数据key
     * @param num 预计数量
     * @param resultList 推荐结果列表
     * @param userId 用户id
     */
    protected void recommendedFilter(KeyGenerator newsPoolKey, Integer num, List<Long> resultList, Long userId) {

        long begin = System.currentTimeMillis();
        //1.获取redis池中的数据
        long total = redisListAdapter.size(newsPoolKey);

        if (0 == total) {
            return;
        }

        List<Long> newsList = redisListAdapter.leftIndex(newsPoolKey, total, Long.class);
        logger.debug("redis拉取列表耗时：{} 列表长：{}", System.currentTimeMillis() - begin, newsList.size());

        //2.去重处理,保存重复数据方便补录推荐记录
        KeyGenerator recommendedNewsKey = RedisKeyConstant.NEWS_NEW_RECOMMENDED.copy().appendKey(userId);
        Set<Long> record = redisSetAdapter.getAllMembers(recommendedNewsKey,Long.class);
        List<Long> repeatNewsList = new ArrayList<>();
        int i =0;
        for (Long id : newsList) {

            //3.判断是否已被推荐过，如果没有则加入结果集并加入推荐记录
            this.generateResult(record, id, resultList,repeatNewsList);
            i=i+1;

            //4.一旦数量达标，则退出循环
            if (resultList.size() >= num) {
                logger.info("[recommendedFilter][redis缓存池]{}推荐器经去重后已取得{}条新闻推荐数据，功成圆满",this.getClass().getSimpleName(), num);
                break;
            }
        }

        //5.对余下的数据进行保存
        redisListAdapter.leftTrim(newsPoolKey, i, total);

        //6.有重复则对重复数据进行补录
        if (repeatNewsList.size() > 0) {
            logger.debug("[recommendedFilter][redis缓存池]布隆过滤器筛选已重复,补录推荐重复新闻：{}", JSON.toJSONString(repeatNewsList));
            newsSyncIntegrationService.generateRecommneded(userId, repeatNewsList);
        }
        logger.debug("[recommendedFilter][redis缓存池]bloom去重耗时：{}", System.currentTimeMillis() - begin);
    }

    /**
     * 对从数据库中查出来的数据进行去重处理
     * @param originList 原数据
     * @param num 预计数量
     * @param resultList 推荐结果列表
     * @param userId 用户id
     */
    protected void recommendedFilter(List<Long> originList, Integer num, List<Long> resultList, Long userId) {

        long begin = System.currentTimeMillis();

        //1.去重处理,保存重复数据方便补录推荐记录
        KeyGenerator recommendedNewsKey = RedisKeyConstant.NEWS_NEW_RECOMMENDED.copy().appendKey(userId);
        Set<Long> record = redisSetAdapter.getAllMembers(recommendedNewsKey,Long.class);
        List<Long> repeatNewsList = new ArrayList<>();
        for (Long id : originList) {

            this.generateResult(record, id, resultList,repeatNewsList);

            if (resultList.size()>=num) {
                logger.info("[recommendedFilter][db数据库]{}推荐器经去重后已取得{}条新闻推荐数据，功成圆满", this.getClass(),num);
                break;
            }
        }

        //2.有重复则进行递归并将重复数据进行补录
        if (repeatNewsList.size() > 0) {
            logger.debug("[recommendedFilter][db数据库]布隆过滤器筛选已重复,推荐重复新闻：{}", JSON.toJSONString(repeatNewsList));
            //推荐出来的有重复说明没有加入推荐记录表中，在这里进行补录
            newsSyncIntegrationService.generateRecommneded(userId, repeatNewsList);
        }

        logger.debug("[recommendedFilter][db数据库]普通去重耗时：{}", System.currentTimeMillis() - begin);
    }


    /**
     * 判断是否已被推荐过:
     * 1.如果没有则加入结果集并加入推荐记录
     * 2.如果有则加入重复推荐列表，便于后续补录
     * @param record 已推荐记录
     * @param id 新闻id
     * @param resultList 推荐结果列表
     * @param repeatList 重复推荐列表
     */
    private void generateResult(Set<Long> record, Long id, List<Long> resultList,List<Long> repeatList) {

        if (!record.contains(id)) {
            resultList.add(id);
            record.add(id);
        }else{
            repeatList.add(id);
        }
    }

}
